#
# Data from Acta Cryst. D62 (2006), 678-682.
#
from __future__ import absolute_import, division, print_function
from iotbx.pdb.amino_acid_codes import three_letter_given_one_letter
import sys

metal_defaults = {
  "Na" : {
    "O" : {
      "HOH" : [2.41, 0.10],
      "ASP" : [2.41, 0.10],
      },
    },
  "Mg" : {
    "O" : {
      "HOH" : [2.07, 0.05],
      "ASP" : [2.07, 0.10],
      },
    },
  "K" : {
    "O" : {
      "HOH" : [2.81, 0.15],
      "ASP" : [2.82, 0.10],
      },
    },
  "Ca" : {
    "O" : {
      "HOH" : [2.39, 0.10],
      "ASP" : [2.36, 0.10],
      },
    },
  "Mn" : {
    "O" : {
      "HOH" : [2.19, 0.05],
      "ASP" : [2.15, 0.05],
      },
    "N" : {
      "HIS" : [2.21, 0.10],
      },
    "S" : {
      "CYS" : [2.35, 0.25],
      },
    },
  "Fe" : {
    "O" : {
      "HOH" : [2.09, 0.10],
      "ASP" : [2.04, 0.10],
      },
    "N" : {
      "HIS" : [2.16, 0.15],
      },
    "S" : {
      "CYS" : [2.30, 0.05],
      },
    },
  "Co" : {
    "O" : {
      "HOH" : [2.09, 0.10],
      "ASP" : [2.05, 0.05],
      },
    "N" : {
      "HIS" : [2.14, 0.10],
      },
    "S" : {
      "CYS" : [2.25, 0.15],
      },
    },
  "Cu" : {
    "O" : {
      "HOH" : [2.13, 0.25],
      "ASP" : [1.99, 0.15],
      },
    "N" : {
      "HIS" : [2.02, 0.10],
      },
    "S" : {
      "CYS" : [2.15, 0.25],
      },
    },
  "Zn" : {
    "O" : {
      "HOH" : [2.09, 0.05],
      "ASP" : [1.99, 0.05],
      },
    "N" : {
      "HIS" : [2.03, 0.05],
      },
    "S" : {
      "CYS" : [2.31, 0.10],
      },
    },
  }

carbonyl = {
  "Na" : [2.38, 0.10],
  "Mg" : [2.26, 0.25],
  "K"  : [2.74, 0.15],
  "Ca" : [2.36, 0.10],
  "Mn" : [2.19, 0.25],
  "Fe" : [2.04, 0.25],
  "Co" : [2.08, 0.25],
  "Cu" : [2.04, 0.25],
  "Zn" : [2.07, 0.25],
  }

for m in metal_defaults:
  for c in metal_defaults[m]:
    if c!="O": continue
    # more waters ?
    for w in ["WAT", "TIP"]:
      metal_defaults[m][c].setdefault(w, [None, None])
      metal_defaults[m][c][w][0] = metal_defaults[m][c]["HOH"][0]
      metal_defaults[m][c][w][1] = metal_defaults[m][c]["HOH"][1]
    # ASP = GLU
    metal_defaults[m][c].setdefault("GLU", [None, None])
    metal_defaults[m][c]["GLU"][0] = metal_defaults[m][c]["ASP"][0]
    metal_defaults[m][c]["GLU"][1] = metal_defaults[m][c]["ASP"][1]
    # ASN, GLN
    metal_defaults[m][c].setdefault("ASN", [None, None])
    metal_defaults[m][c]["ASN"][0] = metal_defaults[m][c]["ASP"][0] + 0.02
    metal_defaults[m][c]["ASN"][1] = metal_defaults[m][c]["ASP"][1]
    metal_defaults[m][c].setdefault("GLN", [None, None])
    metal_defaults[m][c]["GLN"][0] = metal_defaults[m][c]["GLU"][0] + 0.02
    metal_defaults[m][c]["GLN"][1] = metal_defaults[m][c]["GLU"][1]
    # SER, THR
    metal_defaults[m][c].setdefault("SER", [None, None])
    metal_defaults[m][c].setdefault("THR", [None, None])
    metal_defaults[m][c]["SER"][0] = (metal_defaults[m][c]["ASP"][0] + \
                                metal_defaults[m][c]["HOH"][0])/2
    metal_defaults[m][c]["THR"][0] = (metal_defaults[m][c]["ASP"][0] + \
                                metal_defaults[m][c]["HOH"][0])/2
    metal_defaults[m][c]["SER"][1] = metal_defaults[m][c]["ASP"][1]
    metal_defaults[m][c]["THR"][1] = metal_defaults[m][c]["ASP"][1]
    # TYR
    metal_defaults[m][c].setdefault("TYR", [None, None])
    metal_defaults[m][c]["TYR"][0] = metal_defaults[m][c]["ASP"][0] - 0.1
    metal_defaults[m][c]["TYR"][1] = metal_defaults[m][c]["ASP"][1]

non_protein_sigma = 0.25
non_protein = {}
for m in carbonyl:
  non_protein[(m, "O")] = [metal_defaults[m]["O"]["ASN"][0], non_protein_sigma]
  if "N" in metal_defaults[m]:
    non_protein[(m, "N")] = [metal_defaults[m]["N"]["HIS"][0], non_protein_sigma]
  if "S" in metal_defaults[m]:
    non_protein[(m, "S")] = [metal_defaults[m]["S"]["CYS"][0], non_protein_sigma]

qm_defaults = {
    ("H",  "H" )    : 0.579,
    ("H",  "Li")    : 1.629,
    ("H",  "Be")    : 1.334,
    ("H",  "B" )    : 1.115,
    ("H",  "C" )    : 0.970,
    ("H",  "N" )    : 0.907,
    ("H",  "O" )    : 0.885,
    ("H",  "F" )    : 1.037,
    ("H",  "Na")    : 1.907,
    ("H",  "Mg")    : 1.713,
    ("H",  "Al")    : 1.582,
    ("H",  "Si")    : 1.478,
    ("H",  "P" )    : 1.301,
    ("H",  "S" )    : 1.186,
    ("H",  "Cl")    : 1.202,
    ("H",  "K" )    : 2.327,
    ("H",  "Ca")    : 2.120,
    ("H",  "Sc")    : 2.293,
    ("H",  "Ti")    : 2.131,
    ("H",  "V" )    : 2.103,
    ("H",  "Cr")    : 2.146,
    ("H",  "Mn")    : 2.207,
    ("H",  "Fe")    : 2.126,
    ("H",  "Co")    : 2.036,
    ("H",  "Ni")    : 1.972,
    ("H",  "Cu")    : 1.975,
    ("H",  "Ga")    : 1.570,
    ("H",  "Ge")    : 1.527,
    ("H",  "As")    : 1.502,
    ("H",  "Se")    : 1.444,
    ("H",  "Br")    : 1.400,
    ("H",  "Rb")    : 2.208,
    ("H",  "Sr")    : 2.153,
    ("H",  "Y" )    : 1.500,
    ("H",  "Zr")    : 1.500,
    ("H",  "Nb")    : 1.500,
    ("H",  "Mo")    : 1.500,
    ("H",  "Tc")    : 1.500,
    ("H",  "Ru")    : 1.500,
    ("H",  "Rh")    : 1.500,
    ("H",  "Pd")    : 1.500,
    ("H",  "Ag")    : 1.500,
    ("H",  "In")    : 1.577,
    ("H",  "Sn")    : 1.628,
    ("H",  "Sb")    : 1.643,
    ("H",  "Te")    : 1.623,
    ("H",  "I" )    : 1.596,
    ("H",  "Cs")    : 1.500,
    ("H",  "Ba")    : 1.500,
    ("H",  "La")    : 1.500,
    ("H",  "Hf")    : 1.500,
    ("H",  "Ta")    : 1.500,
    ("H",  "W" )    : 1.500,
    ("H",  "Re")    : 1.500,
    ("H",  "Os")    : 1.500,
    ("H",  "Ir")    : 1.500,
    ("H",  "Pt")    : 1.500,
    ("H",  "Au")    : 1.500,
    ("H",  "Tl")    : 1.500,
    ("H",  "Pb")    : 1.500,
    ("H",  "Bi")    : 1.500,
    ("H",  "Po")    : 1.500,
    ("H",  "At")    : 1.500,
    ("Li", "Li")    : 2.806,
    ("Li", "Be")    : 2.471,
    ("Li", "B" )    : 2.235,
    ("Li", "C" )    : 2.004,
    ("Li", "N" )    : 1.749,
    ("Li", "O" )    : 1.590,
    ("Li", "F" )    : 1.555,
    ("Li", "Na")    : 2.998,
    ("Li", "Mg")    : 2.856,
    ("Li", "Al")    : 2.692,
    ("Li", "Si")    : 2.519,
    ("Li", "P" )    : 2.373,
    ("Li", "S" )    : 2.189,
    ("Li", "Cl")    : 2.071,
    ("Li", "K" )    : 3.480,
    ("Li", "Ca")    : 3.328,
    ("Li", "Sc")    : 2.293,
    ("Li", "Ti")    : 2.131,
    ("Li", "V" )    : 2.103,
    ("Li", "Cr")    : 2.146,
    ("Li", "Mn")    : 2.207,
    ("Li", "Fe")    : 2.126,
    ("Li", "Co")    : 2.036,
    ("Li", "Ni")    : 1.972,
    ("Li", "Cu")    : 1.975,
    ("Li", "Ga")    : 2.547,
    ("Li", "Ge")    : 2.472,
    ("Li", "As")    : 2.277,
    ("Li", "Se")    : 2.274,
    ("Li", "Br")    : 2.188,
    ("Li", "Rb")    : 3.224,
    ("Li", "Sr")    : 3.235,
    ("Li", "Y" )    : 2.750,
    ("Li", "Zr")    : 2.750,
    ("Li", "Nb")    : 2.750,
    ("Li", "Mo")    : 2.750,
    ("Li", "Tc")    : 2.750,
    ("Li", "Ru")    : 2.750,
    ("Li", "Rh")    : 2.750,
    ("Li", "Pd")    : 2.750,
    ("Li", "Ag")    : 2.750,
    ("Li", "In")    : 2.688,
    ("Li", "Sn")    : 2.680,
    ("Li", "Sb")    : 2.606,
    ("Li", "Te")    : 2.419,
    ("Li", "I" )    : 2.286,
    ("Li", "Cs")    : 2.750,
    ("Li", "Ba")    : 2.750,
    ("Li", "La")    : 2.750,
    ("Li", "Hf")    : 2.750,
    ("Li", "Ta")    : 2.750,
    ("Li", "W" )    : 2.750,
    ("Li", "Re")    : 2.750,
    ("Li", "Os")    : 2.750,
    ("Li", "Ir")    : 2.750,
    ("Li", "Pt")    : 2.750,
    ("Li", "Au")    : 2.750,
    ("Li", "Tl")    : 2.750,
    ("Li", "Pb")    : 2.750,
    ("Li", "Bi")    : 2.750,
    ("Li", "Po")    : 2.750,
    ("Li", "At")    : 2.750,
    ("Be", "Be")    : 2.121,
    ("Be", "Be", 2) : 4.190,
    ("Be", "B" )    : 1.904,
    ("Be", "B" , 2) : 2.036,
    ("Be", "C" )    : 1.697,
    ("Be", "C" , 2) : 1.541,
    ("Be", "N" )    : 1.501,
    ("Be", "N" , 2) : 1.333,
    ("Be", "O" )    : 1.400,
    ("Be", "O" , 2) : 1.295,
    ("Be", "F" )    : 1.366,
    ("Be", "Na")    : 2.741,
    ("Be", "Mg")    : 2.534,
    ("Be", "Mg", 2) : 4.480,
    ("Be", "Al")    : 2.372,
    ("Be", "Al", 2) : 4.589,
    ("Be", "Si")    : 2.220,
    ("Be", "Si", 2) : 2.341,
    ("Be", "P" )    : 2.074,
    ("Be", "P" , 2) : 1.907,
    ("Be", "S" )    : 1.918,
    ("Be", "S" , 2) : 1.732,
    ("Be", "Cl")    : 1.811,
    ("Be", "K" )    : 3.260,
    ("Be", "Ca")    : 3.029,
    ("Be", "Ca", 2) : 5.008,
    ("Be", "Sc")    : 2.293,
    ("Be", "Ti")    : 2.131,
    ("Be", "Ti", 2) : 2.131,
    ("Be", "V" )    : 2.103,
    ("Be", "V" , 2) : 2.103,
    ("Be", "Cr")    : 2.146,
    ("Be", "Cr", 2) : 2.146,
    ("Be", "Mn")    : 2.207,
    ("Be", "Mn", 2) : 2.207,
    ("Be", "Fe")    : 2.126,
    ("Be", "Fe", 2) : 2.126,
    ("Be", "Co")    : 2.036,
    ("Be", "Co", 2) : 2.036,
    ("Be", "Ni")    : 1.972,
    ("Be", "Ni", 2) : 1.972,
    ("Be", "Cu")    : 1.975,
    ("Be", "Ga")    : 2.280,
    ("Be", "Ga", 2) : 4.413,
    ("Be", "Ge")    : 2.209,
    ("Be", "Ge", 2) : 2.336,
    ("Be", "As")    : 2.137,
    ("Be", "As", 2) : 1.972,
    ("Be", "Se")    : 2.012,
    ("Be", "Se", 2) : 1.829,
    ("Be", "Br")    : 1.938,
    ("Be", "Rb")    : 3.059,
    ("Be", "Sr")    : 3.001,
    ("Be", "Sr", 2) : 3.447,
    ("Be", "Y" )    : 2.750,
    ("Be", "Zr")    : 2.750,
    ("Be", "Zr", 2) : 2.750,
    ("Be", "Nb")    : 2.750,
    ("Be", "Nb", 2) : 2.750,
    ("Be", "Mo")    : 2.750,
    ("Be", "Mo", 2) : 2.750,
    ("Be", "Tc")    : 2.750,
    ("Be", "Tc", 2) : 2.750,
    ("Be", "Ru")    : 2.750,
    ("Be", "Ru", 2) : 2.750,
    ("Be", "Rh")    : 2.750,
    ("Be", "Rh", 2) : 2.750,
    ("Be", "Pd")    : 2.750,
    ("Be", "Pd", 2) : 2.750,
    ("Be", "Ag")    : 2.750,
    ("Be", "In")    : 2.304,
    ("Be", "In", 2) : 3.153,
    ("Be", "Sn")    : 2.363,
    ("Be", "Sn", 2) : 2.275,
    ("Be", "Sb")    : 2.318,
    ("Be", "Sb", 2) : 2.040,
    ("Be", "Te")    : 2.198,
    ("Be", "Te", 2) : 1.952,
    ("Be", "I" )    : 2.104,
    ("Be", "Cs")    : 2.750,
    ("Be", "Ba")    : 2.750,
    ("Be", "Ba", 2) : 2.750,
    ("Be", "La")    : 2.750,
    ("Be", "Hf")    : 2.750,
    ("Be", "Hf", 2) : 2.750,
    ("Be", "Ta")    : 2.750,
    ("Be", "Ta", 2) : 2.750,
    ("Be", "W" )    : 2.750,
    ("Be", "W" , 2) : 2.750,
    ("Be", "Re")    : 2.750,
    ("Be", "Re", 2) : 2.750,
    ("Be", "Os")    : 2.750,
    ("Be", "Os", 2) : 2.750,
    ("Be", "Ir")    : 2.750,
    ("Be", "Ir", 2) : 2.750,
    ("Be", "Pt")    : 2.750,
    ("Be", "Pt", 2) : 2.750,
    ("Be", "Au")    : 2.750,
    ("Be", "Tl")    : 2.750,
    ("Be", "Tl", 2) : 2.750,
    ("Be", "Pb")    : 2.750,
    ("Be", "Pb", 2) : 2.750,
    ("Be", "Bi")    : 2.750,
    ("Be", "Bi", 2) : 2.750,
    ("Be", "Po")    : 2.750,
    ("Be", "Po", 2) : 2.750,
    ("Be", "At")    : 2.750,
    ("B",  "B" )    : 1.778,
    ("B",  "B" , 2) : 1.631,
    ("B",  "C" )    : 1.663,
    ("B",  "C" , 2) : 1.469,
    ("B",  "N" )    : 1.534,
    ("B",  "N" , 2) : 1.394,
    ("B",  "O" )    : 1.440,
    ("B",  "O" , 2) : 1.374,
    ("B",  "F" )    : 1.355,
    ("B",  "Na")    : 2.537,
    ("B",  "Mg")    : 2.319,
    ("B",  "Mg", 2) : 4.210,
    ("B",  "Al")    : 2.154,
    ("B",  "Al", 2) : 2.077,
    ("B",  "Si")    : 2.042,
    ("B",  "Si", 2) : 1.820,
    ("B",  "P" )    : 1.949,
    ("B",  "P" , 2) : 1.855,
    ("B",  "S" )    : 1.909,
    ("B",  "S" , 2) : 1.909,
    ("B",  "Cl")    : 1.820,
    ("B",  "K" )    : 3.001,
    ("B",  "Ca")    : 2.768,
    ("B",  "Ca", 2) : 3.735,
    ("B",  "Sc")    : 2.293,
    ("B",  "Ti")    : 2.131,
    ("B",  "Ti", 2) : 2.131,
    ("B",  "V" )    : 2.103,
    ("B",  "V" , 2) : 2.103,
    ("B",  "Cr")    : 2.146,
    ("B",  "Cr", 2) : 2.146,
    ("B",  "Mn")    : 2.207,
    ("B",  "Mn", 2) : 2.207,
    ("B",  "Fe")    : 2.126,
    ("B",  "Fe", 2) : 2.126,
    ("B",  "Co")    : 2.036,
    ("B",  "Co", 2) : 2.036,
    ("B",  "Ni")    : 1.972,
    ("B",  "Ni", 2) : 1.972,
    ("B",  "Cu")    : 1.975,
    ("B",  "Ga")    : 2.098,
    ("B",  "Ga", 2) : 2.010,
    ("B",  "Ge")    : 2.073,
    ("B",  "Ge", 2) : 1.853,
    ("B",  "As")    : 2.008,
    ("B",  "As", 2) : 1.815,
    ("B",  "Se")    : 1.911,
    ("B",  "Se", 2) : 1.710,
    ("B",  "Br")    : 1.903,
    ("B",  "Rb")    : 2.797,
    ("B",  "Sr")    : 2.762,
    ("B",  "Sr", 2) : 3.096,
    ("B",  "Y" )    : 2.750,
    ("B",  "Zr")    : 2.750,
    ("B",  "Zr", 2) : 2.750,
    ("B",  "Nb")    : 2.750,
    ("B",  "Nb", 2) : 2.750,
    ("B",  "Mo")    : 2.750,
    ("B",  "Mo", 2) : 2.750,
    ("B",  "Tc")    : 2.750,
    ("B",  "Tc", 2) : 2.750,
    ("B",  "Ru")    : 2.750,
    ("B",  "Ru", 2) : 2.750,
    ("B",  "Rh")    : 2.750,
    ("B",  "Rh", 2) : 2.750,
    ("B",  "Pd")    : 2.750,
    ("B",  "Pd", 2) : 2.750,
    ("B",  "Ag")    : 2.750,
    ("B",  "In")    : 2.147,
    ("B",  "In", 2) : 1.742,
    ("B",  "Sn")    : 2.219,
    ("B",  "Sn", 2) : 1.852,
    ("B",  "Sb")    : 2.178,
    ("B",  "Sb", 2) : 1.919,
    ("B",  "Te")    : 2.103,
    ("B",  "Te", 2) : 1.847,
    ("B",  "I" )    : 2.092,
    ("B",  "Cs")    : 2.750,
    ("B",  "Ba")    : 2.750,
    ("B",  "Ba", 2) : 2.750,
    ("B",  "La")    : 2.750,
    ("B",  "Hf")    : 2.750,
    ("B",  "Hf", 2) : 2.750,
    ("B",  "Ta")    : 2.750,
    ("B",  "Ta", 2) : 2.750,
    ("B",  "W" )    : 2.750,
    ("B",  "W" , 2) : 2.750,
    ("B",  "Re")    : 2.750,
    ("B",  "Re", 2) : 2.750,
    ("B",  "Os")    : 2.750,
    ("B",  "Os", 2) : 2.750,
    ("B",  "Ir")    : 2.750,
    ("B",  "Ir", 2) : 2.750,
    ("B",  "Pt")    : 2.750,
    ("B",  "Pt", 2) : 2.750,
    ("B",  "Au")    : 2.750,
    ("B",  "Tl")    : 2.750,
    ("B",  "Tl", 2) : 2.750,
    ("B",  "Pb")    : 2.750,
    ("B",  "Pb", 2) : 2.750,
    ("B",  "Bi")    : 2.750,
    ("B",  "Bi", 2) : 2.750,
    ("B",  "Po")    : 2.750,
    ("B",  "Po", 2) : 2.750,
    ("B",  "At")    : 2.750,
    ("C",  "C" )    : 1.491,
    ("C",  "C" , 2) : 1.365,
    ("C",  "N" )    : 1.430,
    ("C",  "N" , 2) : 1.301,
    ("C",  "O" )    : 1.374,
    ("C",  "O" , 2) : 1.222,
    ("C",  "F" )    : 1.329,
    ("C",  "Na")    : 2.328,
    ("C",  "Mg")    : 2.105,
    ("C",  "Mg", 2) : 2.198,
    ("C",  "Al")    : 1.973,
    ("C",  "Al", 2) : 1.776,
    ("C",  "Si")    : 1.890,
    ("C",  "Si", 2) : 1.695,
    ("C",  "P" )    : 1.826,
    ("C",  "P" , 2) : 1.709,
    ("C",  "S" )    : 1.766,
    ("C",  "S" , 2) : 1.663,
    ("C",  "Cl")    : 1.728,
    ("C",  "K" )    : 2.761,
    ("C",  "Ca")    : 2.519,
    ("C",  "Ca", 2) : 2.627,
    ("C",  "Sc")    : 2.293,
    ("C",  "Ti")    : 2.131,
    ("C",  "Ti", 2) : 2.131,
    ("C",  "V" )    : 2.103,
    ("C",  "V" , 2) : 2.103,
    ("C",  "Cr")    : 2.146,
    ("C",  "Cr", 2) : 2.146,
    ("C",  "Mn")    : 2.207,
    ("C",  "Mn", 2) : 2.207,
    ("C",  "Fe")    : 2.126,
    ("C",  "Fe", 2) : 2.126,
    ("C",  "Co")    : 2.036,
    ("C",  "Co", 2) : 2.036,
    ("C",  "Ni")    : 1.972,
    ("C",  "Ni", 2) : 1.972,
    ("C",  "Cu")    : 1.975,
    ("C",  "Ga")    : 1.977,
    ("C",  "Ga", 2) : 1.779,
    ("C",  "Ge")    : 1.948,
    ("C",  "Ge", 2) : 1.757,
    ("C",  "As")    : 1.963,
    ("C",  "As", 2) : 1.751,
    ("C",  "Se")    : 1.950,
    ("C",  "Se", 2) : 1.724,
    ("C",  "Br")    : 1.941,
    ("C",  "Rb")    : 2.670,
    ("C",  "Sr")    : 2.617,
    ("C",  "Sr", 2) : 2.779,
    ("C",  "Y" )    : 2.750,
    ("C",  "Zr")    : 2.750,
    ("C",  "Zr", 2) : 2.750,
    ("C",  "Nb")    : 2.750,
    ("C",  "Nb", 2) : 2.750,
    ("C",  "Mo")    : 2.750,
    ("C",  "Mo", 2) : 2.750,
    ("C",  "Tc")    : 2.750,
    ("C",  "Tc", 2) : 2.750,
    ("C",  "Ru")    : 2.750,
    ("C",  "Ru", 2) : 2.750,
    ("C",  "Rh")    : 2.750,
    ("C",  "Rh", 2) : 2.750,
    ("C",  "Pd")    : 2.750,
    ("C",  "Pd", 2) : 2.750,
    ("C",  "Ag")    : 2.750,
    ("C",  "In")    : 2.094,
    ("C",  "In", 2) : 1.731,
    ("C",  "Sn")    : 2.111,
    ("C",  "Sn", 2) : 1.824,
    ("C",  "Sb")    : 2.130,
    ("C",  "Sb", 2) : 1.880,
    ("C",  "Te")    : 2.128,
    ("C",  "Te", 2) : 1.890,
    ("C",  "I" )    : 2.110,
    ("C",  "Cs")    : 2.750,
    ("C",  "Ba")    : 2.750,
    ("C",  "Ba", 2) : 2.750,
    ("C",  "La")    : 2.750,
    ("C",  "Hf")    : 2.750,
    ("C",  "Hf", 2) : 2.750,
    ("C",  "Ta")    : 2.750,
    ("C",  "Ta", 2) : 2.750,
    ("C",  "W" )    : 2.750,
    ("C",  "W" , 2) : 2.750,
    ("C",  "Re")    : 2.750,
    ("C",  "Re", 2) : 2.750,
    ("C",  "Os")    : 2.110,
    ("C",  "Os", 2) : 2.110,
    ("C",  "Ir")    : 2.750,
    ("C",  "Ir", 2) : 2.750,
    ("C",  "Pt")    : 2.750,
    ("C",  "Pt", 2) : 2.750,
    ("C",  "Au")    : 2.750,
    ("C",  "Tl")    : 2.750,
    ("C",  "Tl", 2) : 2.750,
    ("C",  "Pb")    : 2.750,
    ("C",  "Pb", 2) : 2.750,
    ("C",  "Bi")    : 2.750,
    ("C",  "Bi", 2) : 2.750,
    ("C",  "Po")    : 2.750,
    ("C",  "Po", 2) : 2.750,
    ("C",  "At")    : 2.750,
    ("N",  "N" )    : 1.350,
    ("N",  "N" , 2) : 1.226,
    ("N",  "O" )    : 1.297,
    ("N",  "O" , 2) : 1.222,
    ("N",  "F" )    : 1.360,
    ("N",  "Na")    : 2.080,
    ("N",  "Mg")    : 1.895,
    ("N",  "Mg", 2) : 1.925,
    ("N",  "Al")    : 1.770,
    ("N",  "Al", 2) : 1.597,
    ("N",  "Si")    : 1.725,
    ("N",  "Si", 2) : 1.576,
    ("N",  "P" )    : 1.653,
    ("N",  "P" , 2) : 1.581,
    ("N",  "S" )    : 1.629,
    ("N",  "S" , 2) : 1.566,
    ("N",  "Cl")    : 1.306,
    ("N",  "K" )    : 2.444,
    ("N",  "Ca")    : 2.250,
    ("N",  "Ca", 2) : 2.275,
    ("N",  "Sc")    : 2.293,
    ("N",  "Ti")    : 2.131,
    ("N",  "Ti", 2) : 2.131,
    ("N",  "V" )    : 2.103,
    ("N",  "V" , 2) : 2.103,
    ("N",  "Cr")    : 2.146,
    ("N",  "Cr", 2) : 2.146,
    ("N",  "Mn")    : 2.207,
    ("N",  "Mn", 2) : 2.207,
    ("N",  "Fe")    : 2.126,
    ("N",  "Fe", 2) : 2.126,
    ("N",  "Co")    : 2.036,
    ("N",  "Co", 2) : 2.036,
    ("N",  "Ni")    : 1.972,
    ("N",  "Ni", 2) : 1.972,
    ("N",  "Cu")    : 1.975,
    ("N",  "Ga")    : 1.816,
    ("N",  "Ga", 2) : 1.679,
    ("N",  "Ge")    : 1.830,
    ("N",  "Ge", 2) : 1.683,
    ("N",  "As")    : 1.834,
    ("N",  "As", 2) : 1.676,
    ("N",  "Se")    : 1.842,
    ("N",  "Se", 2) : 1.673,
    ("N",  "Br")    : 1.878,
    ("N",  "Rb")    : 2.565,
    ("N",  "Sr")    : 2.493,
    ("N",  "Sr", 2) : 2.631,
    ("N",  "Y" )    : 2.750,
    ("N",  "Zr")    : 2.750,
    ("N",  "Zr", 2) : 2.750,
    ("N",  "Nb")    : 2.750,
    ("N",  "Nb", 2) : 2.750,
    ("N",  "Mo")    : 2.750,
    ("N",  "Mo", 2) : 2.750,
    ("N",  "Tc")    : 2.750,
    ("N",  "Tc", 2) : 2.750,
    ("N",  "Ru")    : 2.750,
    ("N",  "Ru", 2) : 2.750,
    ("N",  "Rh")    : 2.750,
    ("N",  "Rh", 2) : 2.750,
    ("N",  "Pd")    : 2.750,
    ("N",  "Pd", 2) : 2.750,
    ("N",  "Ag")    : 2.750,
    ("N",  "In")    : 1.864,
    ("N",  "In", 2) : 1.622,
    ("N",  "Sn")    : 1.979,
    ("N",  "Sn", 2) : 1.811,
    ("N",  "Sb")    : 2.004,
    ("N",  "Sb", 2) : 1.852,
    ("N",  "Te")    : 2.045,
    ("N",  "Te", 2) : 1.888,
    ("N",  "I" )    : 2.070,
    ("N",  "Cs")    : 2.750,
    ("N",  "Ba")    : 2.750,
    ("N",  "Ba", 2) : 2.750,
    ("N",  "La")    : 2.750,
    ("N",  "Hf")    : 2.750,
    ("N",  "Hf", 2) : 2.750,
    ("N",  "Ta")    : 2.750,
    ("N",  "Ta", 2) : 2.750,
    ("N",  "W" )    : 2.750,
    ("N",  "W" , 2) : 2.750,
    ("N",  "Re")    : 2.750,
    ("N",  "Re", 2) : 2.750,
    ("N",  "Os")    : 2.110,
    ("N",  "Os", 2) : 2.110,
    ("N",  "Ir")    : 2.750,
    ("N",  "Ir", 2) : 2.750,
    ("N",  "Pt")    : 2.750,
    ("N",  "Pt", 2) : 2.750,
    ("N",  "Au")    : 2.750,
    ("N",  "Tl")    : 2.750,
    ("N",  "Tl", 2) : 2.750,
    ("N",  "Pb")    : 2.750,
    ("N",  "Pb", 2) : 2.750,
    ("N",  "Bi")    : 2.750,
    ("N",  "Bi", 2) : 2.750,
    ("N",  "Po")    : 2.750,
    ("N",  "Po", 2) : 2.750,
    ("N",  "At")    : 2.750,
    ("O",  "O" )    : 1.262,
    ("O",  "O" , 2) : 1.270,
    ("O",  "F" )    : 1.128,
    ("O",  "Na")    : 1.920,
    ("O",  "Mg")    : 1.754,
    ("O",  "Mg", 2) : 1.739,
    ("O",  "Al")    : 1.693,
    ("O",  "Al", 2) : 1.572,
    ("O",  "Si")    : 1.650,
    ("O",  "Si", 2) : 1.502,
    ("O",  "P" )    : 1.558,
    ("O",  "P" , 2) : 1.497,
    ("O",  "S" )    : 1.466,
    ("O",  "S" , 2) : 1.443,
    ("O",  "Cl")    : 1.400,
    ("O",  "K" )    : 2.242,
    ("O",  "Ca")    : 2.077,
    ("O",  "Ca", 2) : 2.038,
    ("O",  "Sc")    : 2.293,
    ("O",  "Ti")    : 2.131,
    ("O",  "Ti", 2) : 2.131,
    ("O",  "V" )    : 2.103,
    ("O",  "V" , 2) : 2.103,
    ("O",  "Cr")    : 2.146,
    ("O",  "Cr", 2) : 2.146,
    ("O",  "Mn")    : 2.207,
    ("O",  "Mn", 2) : 2.207,
    ("O",  "Fe")    : 2.126,
    ("O",  "Fe", 2) : 2.126,
    ("O",  "Co")    : 2.036,
    ("O",  "Co", 2) : 2.036,
    ("O",  "Ni")    : 1.972,
    ("O",  "Ni", 2) : 1.972,
    ("O",  "Cu")    : 1.975,
    ("O",  "Ga")    : 1.781,
    ("O",  "Ga", 2) : 1.633,
    ("O",  "Ge")    : 1.769,
    ("O",  "Ge", 2) : 1.618,
    ("O",  "As")    : 1.778,
    ("O",  "As", 2) : 1.600,
    ("O",  "Se")    : 1.781,
    ("O",  "Se", 2) : 1.606,
    ("O",  "Br")    : 1.803,
    ("O",  "Rb")    : 2.450,
    ("O",  "Sr")    : 2.400,
    ("O",  "Sr", 2) : 3.765,
    ("O",  "Y" )    : 2.750,
    ("O",  "Zr")    : 2.750,
    ("O",  "Zr", 2) : 2.750,
    ("O",  "Nb")    : 2.750,
    ("O",  "Nb", 2) : 2.750,
    ("O",  "Mo")    : 2.750,
    ("O",  "Mo", 2) : 2.750,
    ("O",  "Tc")    : 2.750,
    ("O",  "Tc", 2) : 2.750,
    ("O",  "Ru")    : 2.750,
    ("O",  "Ru", 2) : 2.750,
    ("O",  "Rh")    : 2.750,
    ("O",  "Rh", 2) : 2.750,
    ("O",  "Pd")    : 2.750,
    ("O",  "Pd", 2) : 2.750,
    ("O",  "Ag")    : 2.750,
    ("O",  "In")    : 1.899,
    ("O",  "In", 2) : 1.683,
    ("O",  "Sn")    : 1.955,
    ("O",  "Sn", 2) : 1.778,
    ("O",  "Sb")    : 1.979,
    ("O",  "Sb", 2) : 1.804,
    ("O",  "Te")    : 2.024,
    ("O",  "Te", 2) : 1.849,
    ("O",  "I" )    : 2.019,
    ("O",  "Cs")    : 2.750,
    ("O",  "Ba")    : 2.750,
    ("O",  "Ba", 2) : 2.750,
    ("O",  "La")    : 2.750,
    ("O",  "Hf")    : 2.750,
    ("O",  "Hf", 2) : 2.750,
    ("O",  "Ta")    : 2.750,
    ("O",  "Ta", 2) : 2.750,
    ("O",  "W" )    : 2.750,
    ("O",  "W" , 2) : 2.750,
    ("O",  "Re")    : 2.750,
    ("O",  "Re", 2) : 2.750,
    ("O",  "Os")    : 2.110,
    ("O",  "Os", 2) : 2.110,
    ("O",  "Ir")    : 2.750,
    ("O",  "Ir", 2) : 2.750,
    ("O",  "Pt")    : 2.750,
    ("O",  "Pt", 2) : 2.750,
    ("O",  "Au")    : 2.750,
    ("O",  "Tl")    : 2.750,
    ("O",  "Tl", 2) : 2.750,
    ("O",  "Pb")    : 2.750,
    ("O",  "Pb", 2) : 2.750,
    ("O",  "Bi")    : 2.750,
    ("O",  "Bi", 2) : 2.750,
    ("O",  "Po")    : 2.750,
    ("O",  "Po", 2) : 2.750,
    ("O",  "At")    : 2.750,
    ("F",  "F" )    : 1.179,
    ("F",  "Na")    : 1.886,
    ("F",  "Mg")    : 1.730,
    ("F",  "Al")    : 1.640,
    ("F",  "Si")    : 1.601,
    ("F",  "P" )    : 1.559,
    ("F",  "S" )    : 1.548,
    ("F",  "Cl")    : 1.369,
    ("F",  "K" )    : 2.189,
    ("F",  "Ca")    : 2.043,
    ("F",  "Sc")    : 2.293,
    ("F",  "Ti")    : 2.131,
    ("F",  "V" )    : 2.103,
    ("F",  "Cr")    : 2.146,
    ("F",  "Mn")    : 2.207,
    ("F",  "Fe")    : 2.126,
    ("F",  "Co")    : 2.036,
    ("F",  "Ni")    : 1.972,
    ("F",  "Cu")    : 1.975,
    ("F",  "Ga")    : 1.741,
    ("F",  "Ge")    : 1.723,
    ("F",  "As")    : 1.724,
    ("F",  "Se")    : 1.730,
    ("F",  "Br")    : 1.735,
    ("F",  "Rb")    : 2.363,
    ("F",  "Sr")    : 2.323,
    ("F",  "Y" )    : 2.750,
    ("F",  "Zr")    : 2.750,
    ("F",  "Nb")    : 2.750,
    ("F",  "Mo")    : 2.750,
    ("F",  "Tc")    : 2.750,
    ("F",  "Ru")    : 2.750,
    ("F",  "Rh")    : 2.750,
    ("F",  "Pd")    : 2.750,
    ("F",  "Ag")    : 2.750,
    ("F",  "In")    : 1.874,
    ("F",  "Sn")    : 1.899,
    ("F",  "Sb")    : 1.928,
    ("F",  "Te")    : 1.959,
    ("F",  "I" )    : 1.956,
    ("F",  "Cs")    : 2.750,
    ("F",  "Ba")    : 2.750,
    ("F",  "La")    : 2.750,
    ("F",  "Hf")    : 2.750,
    ("F",  "Ta")    : 2.750,
    ("F",  "W" )    : 2.750,
    ("F",  "Re")    : 2.750,
    ("F",  "Os")    : 2.750,
    ("F",  "Ir")    : 2.750,
    ("F",  "Pt")    : 2.750,
    ("F",  "Au")    : 2.750,
    ("F",  "Tl")    : 2.750,
    ("F",  "Pb")    : 2.750,
    ("F",  "Bi")    : 2.750,
    ("F",  "Po")    : 2.750,
    ("F",  "At")    : 2.750,
    ("Na", "Na")    : 3.164,
    ("Na", "Mg")    : 3.094,
    ("Na", "Al")    : 2.954,
    ("Na", "Si")    : 2.814,
    ("Na", "P" )    : 2.551,
    ("Na", "S" )    : 2.514,
    ("Na", "Cl")    : 2.397,
    ("Na", "K" )    : 3.719,
    ("Na", "Ca")    : 3.552,
    ("Na", "Sc")    : 2.293,
    ("Na", "Ti")    : 2.131,
    ("Na", "V" )    : 2.103,
    ("Na", "Cr")    : 2.146,
    ("Na", "Mn")    : 2.207,
    ("Na", "Fe")    : 2.126,
    ("Na", "Co")    : 2.036,
    ("Na", "Ni")    : 1.972,
    ("Na", "Cu")    : 1.975,
    ("Na", "Ga")    : 2.823,
    ("Na", "Ge")    : 2.765,
    ("Na", "As")    : 2.590,
    ("Na", "Se")    : 2.599,
    ("Na", "Br")    : 2.512,
    ("Na", "Rb")    : 3.172,
    ("Na", "Sr")    : 3.221,
    ("Na", "Y" )    : 2.750,
    ("Na", "Zr")    : 2.750,
    ("Na", "Nb")    : 2.750,
    ("Na", "Mo")    : 2.750,
    ("Na", "Tc")    : 2.750,
    ("Na", "Ru")    : 2.750,
    ("Na", "Rh")    : 2.750,
    ("Na", "Pd")    : 2.750,
    ("Na", "Ag")    : 2.750,
    ("Na", "In")    : 2.774,
    ("Na", "Sn")    : 2.785,
    ("Na", "Sb")    : 2.781,
    ("Na", "Te")    : 2.699,
    ("Na", "I" )    : 2.601,
    ("Na", "Cs")    : 2.750,
    ("Na", "Ba")    : 2.750,
    ("Na", "La")    : 2.750,
    ("Na", "Hf")    : 2.750,
    ("Na", "Ta")    : 2.750,
    ("Na", "W" )    : 2.750,
    ("Na", "Re")    : 2.750,
    ("Na", "Os")    : 2.750,
    ("Na", "Ir")    : 2.750,
    ("Na", "Pt")    : 2.750,
    ("Na", "Au")    : 2.750,
    ("Na", "Tl")    : 2.750,
    ("Na", "Pb")    : 2.750,
    ("Na", "Bi")    : 2.750,
    ("Na", "Po")    : 2.750,
    ("Na", "At")    : 2.750,
    ("Mg", "Mg")    : 2.915,
    ("Mg", "Mg", 2) : 5.197,
    ("Mg", "Al")    : 2.771,
    ("Mg", "Al", 2) : 4.774,
    ("Mg", "Si")    : 2.618,
    ("Mg", "Si", 2) : 2.820,
    ("Mg", "P" )    : 2.478,
    ("Mg", "P" , 2) : 2.359,
    ("Mg", "S" )    : 2.318,
    ("Mg", "S" , 2) : 2.149,
    ("Mg", "Cl")    : 2.213,
    ("Mg", "K" )    : 3.626,
    ("Mg", "Ca")    : 3.388,
    ("Mg", "Ca", 2) : 5.607,
    ("Mg", "Sc")    : 2.293,
    ("Mg", "Ti")    : 2.131,
    ("Mg", "Ti", 2) : 2.131,
    ("Mg", "V" )    : 2.103,
    ("Mg", "V" , 2) : 2.103,
    ("Mg", "Cr")    : 2.146,
    ("Mg", "Cr", 2) : 2.146,
    ("Mg", "Mn")    : 2.207,
    ("Mg", "Mn", 2) : 2.207,
    ("Mg", "Fe")    : 2.126,
    ("Mg", "Fe", 2) : 2.126,
    ("Mg", "Co")    : 2.036,
    ("Mg", "Co", 2) : 2.036,
    ("Mg", "Ni")    : 1.972,
    ("Mg", "Ni", 2) : 1.972,
    ("Mg", "Cu")    : 1.975,
    ("Mg", "Ga")    : 2.648,
    ("Mg", "Ga", 2) : 4.437,
    ("Mg", "Ge")    : 2.587,
    ("Mg", "Ge", 2) : 2.729,
    ("Mg", "As")    : 2.535,
    ("Mg", "As", 2) : 2.403,
    ("Mg", "Se")    : 2.413,
    ("Mg", "Se", 2) : 2.240,
    ("Mg", "Br")    : 2.336,
    ("Mg", "Rb")    : 3.327,
    ("Mg", "Sr")    : 3.292,
    ("Mg", "Sr", 2) : 3.684,
    ("Mg", "Y" )    : 2.750,
    ("Mg", "Zr")    : 2.750,
    ("Mg", "Zr", 2) : 2.750,
    ("Mg", "Nb")    : 2.750,
    ("Mg", "Nb", 2) : 2.750,
    ("Mg", "Mo")    : 2.750,
    ("Mg", "Mo", 2) : 2.750,
    ("Mg", "Tc")    : 2.750,
    ("Mg", "Tc", 2) : 2.750,
    ("Mg", "Ru")    : 2.750,
    ("Mg", "Ru", 2) : 2.750,
    ("Mg", "Rh")    : 2.750,
    ("Mg", "Rh", 2) : 2.750,
    ("Mg", "Pd")    : 2.750,
    ("Mg", "Pd", 2) : 2.750,
    ("Mg", "Ag")    : 2.750,
    ("Mg", "In")    : 2.756,
    ("Mg", "In", 2) : 3.401,
    ("Mg", "Sn")    : 2.740,
    ("Mg", "Sn", 2) : 2.960,
    ("Mg", "Sb")    : 2.710,
    ("Mg", "Sb", 2) : 2.555,
    ("Mg", "Te")    : 2.603,
    ("Mg", "Te", 2) : 2.410,
    ("Mg", "I" )    : 2.503,
    ("Mg", "Cs")    : 2.750,
    ("Mg", "Ba")    : 2.750,
    ("Mg", "Ba", 2) : 2.750,
    ("Mg", "La")    : 2.750,
    ("Mg", "Hf")    : 2.750,
    ("Mg", "Hf", 2) : 2.750,
    ("Mg", "Ta")    : 2.750,
    ("Mg", "Ta", 2) : 2.750,
    ("Mg", "W" )    : 2.750,
    ("Mg", "W" , 2) : 2.750,
    ("Mg", "Re")    : 2.750,
    ("Mg", "Re", 2) : 2.750,
    ("Mg", "Os")    : 2.750,
    ("Mg", "Os", 2) : 2.750,
    ("Mg", "Ir")    : 2.750,
    ("Mg", "Ir", 2) : 2.750,
    ("Mg", "Pt")    : 2.750,
    ("Mg", "Pt", 2) : 2.750,
    ("Mg", "Au")    : 2.750,
    ("Mg", "Tl")    : 2.750,
    ("Mg", "Tl", 2) : 2.750,
    ("Mg", "Pb")    : 2.750,
    ("Mg", "Pb", 2) : 2.750,
    ("Mg", "Bi")    : 2.750,
    ("Mg", "Bi", 2) : 2.750,
    ("Mg", "Po")    : 2.750,
    ("Mg", "Po", 2) : 2.750,
    ("Mg", "At")    : 2.750,
    ("Al", "Al")    : 2.615,
    ("Al", "Al", 2) : 2.763,
    ("Al", "Si")    : 2.480,
    ("Al", "Si", 2) : 2.382,
    ("Al", "P" )    : 2.368,
    ("Al", "P" , 2) : 2.144,
    ("Al", "S" )    : 2.195,
    ("Al", "S" , 2) : 1.993,
    ("Al", "Cl")    : 2.111,
    ("Al", "K" )    : 3.476,
    ("Al", "Ca")    : 3.247,
    ("Al", "Ca", 2) : 4.995,
    ("Al", "Sc")    : 2.293,
    ("Al", "Ti")    : 2.131,
    ("Al", "Ti", 2) : 2.131,
    ("Al", "V" )    : 2.103,
    ("Al", "V" , 2) : 2.103,
    ("Al", "Cr")    : 2.146,
    ("Al", "Cr", 2) : 2.146,
    ("Al", "Mn")    : 2.207,
    ("Al", "Mn", 2) : 2.207,
    ("Al", "Fe")    : 2.126,
    ("Al", "Fe", 2) : 2.126,
    ("Al", "Co")    : 2.036,
    ("Al", "Co", 2) : 2.036,
    ("Al", "Ni")    : 1.972,
    ("Al", "Ni", 2) : 1.972,
    ("Al", "Cu")    : 1.975,
    ("Al", "Ga")    : 2.529,
    ("Al", "Ga", 2) : 2.636,
    ("Al", "Ge")    : 2.476,
    ("Al", "Ge", 2) : 2.389,
    ("Al", "As")    : 2.443,
    ("Al", "As", 2) : 2.213,
    ("Al", "Se")    : 2.300,
    ("Al", "Se", 2) : 2.094,
    ("Al", "Br")    : 2.249,
    ("Al", "Rb")    : 3.277,
    ("Al", "Sr")    : 3.216,
    ("Al", "Sr", 2) : 3.337,
    ("Al", "Y" )    : 2.750,
    ("Al", "Zr")    : 2.750,
    ("Al", "Zr", 2) : 2.750,
    ("Al", "Nb")    : 2.750,
    ("Al", "Nb", 2) : 2.750,
    ("Al", "Mo")    : 2.750,
    ("Al", "Mo", 2) : 2.750,
    ("Al", "Tc")    : 2.750,
    ("Al", "Tc", 2) : 2.750,
    ("Al", "Ru")    : 2.750,
    ("Al", "Ru", 2) : 2.750,
    ("Al", "Rh")    : 2.750,
    ("Al", "Rh", 2) : 2.750,
    ("Al", "Pd")    : 2.750,
    ("Al", "Pd", 2) : 2.750,
    ("Al", "Ag")    : 2.750,
    ("Al", "In")    : 2.651,
    ("Al", "In", 2) : 2.620,
    ("Al", "Sn")    : 2.667,
    ("Al", "Sn", 2) : 2.515,
    ("Al", "Sb")    : 2.662,
    ("Al", "Sb", 2) : 2.371,
    ("Al", "Te")    : 2.518,
    ("Al", "Te", 2) : 2.279,
    ("Al", "I" )    : 2.475,
    ("Al", "Cs")    : 2.750,
    ("Al", "Ba")    : 2.750,
    ("Al", "Ba", 2) : 2.750,
    ("Al", "La")    : 2.750,
    ("Al", "Hf")    : 2.750,
    ("Al", "Hf", 2) : 2.750,
    ("Al", "Ta")    : 2.750,
    ("Al", "Ta", 2) : 2.750,
    ("Al", "W" )    : 2.750,
    ("Al", "W" , 2) : 2.750,
    ("Al", "Re")    : 2.750,
    ("Al", "Re", 2) : 2.750,
    ("Al", "Os")    : 2.750,
    ("Al", "Os", 2) : 2.750,
    ("Al", "Ir")    : 2.750,
    ("Al", "Ir", 2) : 2.750,
    ("Al", "Pt")    : 2.750,
    ("Al", "Pt", 2) : 2.750,
    ("Al", "Au")    : 2.750,
    ("Al", "Tl")    : 2.750,
    ("Al", "Tl", 2) : 2.750,
    ("Al", "Pb")    : 2.750,
    ("Al", "Pb", 2) : 2.750,
    ("Al", "Bi")    : 2.750,
    ("Al", "Bi", 2) : 2.750,
    ("Al", "Po")    : 2.750,
    ("Al", "Po", 2) : 2.750,
    ("Al", "At")    : 2.750,
    ("Si", "Si")    : 2.362,
    ("Si", "Si", 2) : 2.130,
    ("Si", "P" )    : 2.283,
    ("Si", "P" , 2) : 2.063,
    ("Si", "S" )    : 2.156,
    ("Si", "S" , 2) : 1.938,
    ("Si", "Cl")    : 2.074,
    ("Si", "K" )    : 3.290,
    ("Si", "Ca")    : 3.068,
    ("Si", "Ca", 2) : 3.249,
    ("Si", "Sc")    : 2.293,
    ("Si", "Ti")    : 2.131,
    ("Si", "Ti", 2) : 2.131,
    ("Si", "V" )    : 2.103,
    ("Si", "V" , 2) : 2.103,
    ("Si", "Cr")    : 2.146,
    ("Si", "Cr", 2) : 2.146,
    ("Si", "Mn")    : 2.207,
    ("Si", "Mn", 2) : 2.207,
    ("Si", "Fe")    : 2.126,
    ("Si", "Fe", 2) : 2.126,
    ("Si", "Co")    : 2.036,
    ("Si", "Co", 2) : 2.036,
    ("Si", "Ni")    : 1.972,
    ("Si", "Ni", 2) : 1.972,
    ("Si", "Cu")    : 1.975,
    ("Si", "Ga")    : 2.435,
    ("Si", "Ga", 2) : 2.354,
    ("Si", "Ge")    : 2.386,
    ("Si", "Ge", 2) : 2.151,
    ("Si", "As")    : 2.361,
    ("Si", "As", 2) : 2.142,
    ("Si", "Se")    : 2.274,
    ("Si", "Se", 2) : 2.052,
    ("Si", "Br")    : 2.221,
    ("Si", "Rb")    : 3.167,
    ("Si", "Sr")    : 3.092,
    ("Si", "Sr", 2) : 3.070,
    ("Si", "Y" )    : 2.750,
    ("Si", "Zr")    : 2.750,
    ("Si", "Zr", 2) : 2.750,
    ("Si", "Nb")    : 2.750,
    ("Si", "Nb", 2) : 2.750,
    ("Si", "Mo")    : 2.750,
    ("Si", "Mo", 2) : 2.750,
    ("Si", "Tc")    : 2.750,
    ("Si", "Tc", 2) : 2.750,
    ("Si", "Ru")    : 2.750,
    ("Si", "Ru", 2) : 2.750,
    ("Si", "Rh")    : 2.750,
    ("Si", "Rh", 2) : 2.750,
    ("Si", "Pd")    : 2.750,
    ("Si", "Pd", 2) : 2.750,
    ("Si", "Ag")    : 2.750,
    ("Si", "In")    : 2.570,
    ("Si", "In", 2) : 2.334,
    ("Si", "Sn")    : 2.571,
    ("Si", "Sn", 2) : 2.296,
    ("Si", "Sb")    : 2.578,
    ("Si", "Sb", 2) : 2.318,
    ("Si", "Te")    : 2.531,
    ("Si", "Te", 2) : 2.289,
    ("Si", "I" )    : 2.485,
    ("Si", "Cs")    : 2.750,
    ("Si", "Ba")    : 2.750,
    ("Si", "Ba", 2) : 2.750,
    ("Si", "La")    : 2.750,
    ("Si", "Hf")    : 2.750,
    ("Si", "Hf", 2) : 2.750,
    ("Si", "Ta")    : 2.750,
    ("Si", "Ta", 2) : 2.750,
    ("Si", "W" )    : 2.750,
    ("Si", "W" , 2) : 2.750,
    ("Si", "Re")    : 2.750,
    ("Si", "Re", 2) : 2.750,
    ("Si", "Os")    : 2.750,
    ("Si", "Os", 2) : 2.750,
    ("Si", "Ir")    : 2.750,
    ("Si", "Ir", 2) : 2.750,
    ("Si", "Pt")    : 2.750,
    ("Si", "Pt", 2) : 2.750,
    ("Si", "Au")    : 2.750,
    ("Si", "Tl")    : 2.750,
    ("Si", "Tl", 2) : 2.750,
    ("Si", "Pb")    : 2.750,
    ("Si", "Pb", 2) : 2.750,
    ("Si", "Bi")    : 2.750,
    ("Si", "Bi", 2) : 2.750,
    ("Si", "Po")    : 2.750,
    ("Si", "Po", 2) : 2.750,
    ("Si", "At")    : 2.750,
    ("P",  "P" )    : 2.148,
    ("P",  "P" , 2) : 2.086,
    ("P",  "S" )    : 2.021,
    ("P",  "S" , 2) : 1.950,
    ("P",  "Cl")    : 2.020,
    ("P",  "K" )    : 2.969,
    ("P",  "Ca")    : 2.911,
    ("P",  "Ca", 2) : 2.733,
    ("P",  "Sc")    : 2.293,
    ("P",  "Ti")    : 2.131,
    ("P",  "Ti", 2) : 2.131,
    ("P",  "V" )    : 2.103,
    ("P",  "V" , 2) : 2.103,
    ("P",  "Cr")    : 2.146,
    ("P",  "Cr", 2) : 2.146,
    ("P",  "Mn")    : 2.207,
    ("P",  "Mn", 2) : 2.207,
    ("P",  "Fe")    : 2.126,
    ("P",  "Fe", 2) : 2.126,
    ("P",  "Co")    : 2.036,
    ("P",  "Co", 2) : 2.036,
    ("P",  "Ni")    : 1.972,
    ("P",  "Ni", 2) : 1.972,
    ("P",  "Cu")    : 1.975,
    ("P",  "Ga")    : 2.362,
    ("P",  "Ga", 2) : 2.133,
    ("P",  "Ge")    : 2.330,
    ("P",  "Ge", 2) : 2.112,
    ("P",  "As")    : 2.326,
    ("P",  "As", 2) : 2.110,
    ("P",  "Se")    : 2.257,
    ("P",  "Se", 2) : 2.043,
    ("P",  "Br")    : 2.225,
    ("P",  "Rb")    : 3.038,
    ("P",  "Sr")    : 2.971,
    ("P",  "Sr", 2) : 2.871,
    ("P",  "Y" )    : 2.750,
    ("P",  "Zr")    : 2.750,
    ("P",  "Zr", 2) : 2.750,
    ("P",  "Nb")    : 2.750,
    ("P",  "Nb", 2) : 2.750,
    ("P",  "Mo")    : 2.750,
    ("P",  "Mo", 2) : 2.750,
    ("P",  "Tc")    : 2.750,
    ("P",  "Tc", 2) : 2.750,
    ("P",  "Ru")    : 2.750,
    ("P",  "Ru", 2) : 2.750,
    ("P",  "Rh")    : 2.750,
    ("P",  "Rh", 2) : 2.750,
    ("P",  "Pd")    : 2.750,
    ("P",  "Pd", 2) : 2.750,
    ("P",  "Ag")    : 2.750,
    ("P",  "In")    : 2.508,
    ("P",  "In", 2) : 2.183,
    ("P",  "Sn")    : 2.507,
    ("P",  "Sn", 2) : 2.259,
    ("P",  "Sb")    : 2.523,
    ("P",  "Sb", 2) : 2.523,
    ("P",  "Te")    : 2.498,
    ("P",  "Te", 2) : 2.296,
    ("P",  "I" )    : 2.478,
    ("P",  "Cs")    : 2.750,
    ("P",  "Ba")    : 2.750,
    ("P",  "Ba", 2) : 2.750,
    ("P",  "La")    : 2.750,
    ("P",  "Hf")    : 2.750,
    ("P",  "Hf", 2) : 2.750,
    ("P",  "Ta")    : 2.750,
    ("P",  "Ta", 2) : 2.750,
    ("P",  "W" )    : 2.750,
    ("P",  "W" , 2) : 2.750,
    ("P",  "Re")    : 2.750,
    ("P",  "Re", 2) : 2.750,
    ("P",  "Os")    : 2.750,
    ("P",  "Os", 2) : 2.750,
    ("P",  "Ir")    : 2.750,
    ("P",  "Ir", 2) : 2.750,
    ("P",  "Pt")    : 2.750,
    ("P",  "Pt", 2) : 2.750,
    ("P",  "Au")    : 2.750,
    ("P",  "Tl")    : 2.750,
    ("P",  "Tl", 2) : 2.750,
    ("P",  "Pb")    : 2.750,
    ("P",  "Pb", 2) : 2.750,
    ("P",  "Bi")    : 2.750,
    ("P",  "Bi", 2) : 2.750,
    ("P",  "Po")    : 2.750,
    ("P",  "Po", 2) : 2.750,
    ("P",  "At")    : 2.750,
    ("S",  "S" )    : 2.011,
    ("S",  "S" , 2) : 1.986,
    ("S",  "Cl")    : 2.110,
    ("S",  "K" )    : 2.923,
    ("S",  "Ca")    : 2.712,
    ("S",  "Ca", 2) : 2.486,
    ("S",  "Sc")    : 2.293,
    ("S",  "Ti")    : 2.131,
    ("S",  "Ti", 2) : 2.131,
    ("S",  "V" )    : 2.103,
    ("S",  "V" , 2) : 2.103,
    ("S",  "Cr")    : 2.146,
    ("S",  "Cr", 2) : 2.146,
    ("S",  "Mn")    : 2.207,
    ("S",  "Mn", 2) : 2.207,
    ("S",  "Fe")    : 2.126,
    ("S",  "Fe", 2) : 2.126,
    ("S",  "Co")    : 2.036,
    ("S",  "Co", 2) : 2.036,
    ("S",  "Ni")    : 1.972,
    ("S",  "Ni", 2) : 1.972,
    ("S",  "Cu")    : 1.975,
    ("S",  "Ga")    : 2.206,
    ("S",  "Ga", 2) : 2.008,
    ("S",  "Ge")    : 2.233,
    ("S",  "Ge", 2) : 2.012,
    ("S",  "As")    : 2.245,
    ("S",  "As", 2) : 2.023,
    ("S",  "Se")    : 2.233,
    ("S",  "Se", 2) : 2.011,
    ("S",  "Br")    : 2.182,
    ("S",  "Rb")    : 2.903,
    ("S",  "Sr")    : 2.831,
    ("S",  "Sr", 2) : 2.705,
    ("S",  "Y" )    : 2.750,
    ("S",  "Zr")    : 2.750,
    ("S",  "Zr", 2) : 2.750,
    ("S",  "Nb")    : 2.750,
    ("S",  "Nb", 2) : 2.750,
    ("S",  "Mo")    : 2.750,
    ("S",  "Mo", 2) : 2.750,
    ("S",  "Tc")    : 2.750,
    ("S",  "Tc", 2) : 2.750,
    ("S",  "Ru")    : 2.750,
    ("S",  "Ru", 2) : 2.750,
    ("S",  "Rh")    : 2.750,
    ("S",  "Rh", 2) : 2.750,
    ("S",  "Pd")    : 2.750,
    ("S",  "Pd", 2) : 2.750,
    ("S",  "Ag")    : 2.750,
    ("S",  "In")    : 2.348,
    ("S",  "In", 2) : 2.065,
    ("S",  "Sn")    : 2.401,
    ("S",  "Sn", 2) : 2.165,
    ("S",  "Sb")    : 2.429,
    ("S",  "Sb", 2) : 2.216,
    ("S",  "Te")    : 2.425,
    ("S",  "Te", 2) : 2.252,
    ("S",  "I" )    : 2.421,
    ("S",  "Cs")    : 2.750,
    ("S",  "Ba")    : 2.750,
    ("S",  "Ba", 2) : 2.750,
    ("S",  "La")    : 2.750,
    ("S",  "Hf")    : 2.750,
    ("S",  "Hf", 2) : 2.750,
    ("S",  "Ta")    : 2.750,
    ("S",  "Ta", 2) : 2.750,
    ("S",  "W" )    : 2.750,
    ("S",  "W" , 2) : 2.750,
    ("S",  "Re")    : 2.750,
    ("S",  "Re", 2) : 2.750,
    ("S",  "Os")    : 2.750,
    ("S",  "Os", 2) : 2.750,
    ("S",  "Ir")    : 2.750,
    ("S",  "Ir", 2) : 2.750,
    ("S",  "Pt")    : 2.750,
    ("S",  "Pt", 2) : 2.750,
    ("S",  "Au")    : 2.750,
    ("S",  "Tl")    : 2.750,
    ("S",  "Tl", 2) : 2.750,
    ("S",  "Pb")    : 2.750,
    ("S",  "Pb", 2) : 2.750,
    ("S",  "Bi")    : 2.750,
    ("S",  "Bi", 2) : 2.750,
    ("S",  "Po")    : 2.750,
    ("S",  "Po", 2) : 2.750,
    ("S",  "At")    : 2.750,
    ("Cl", "Cl")    : 1.410,
    ("Cl", "K" )    : 2.775,
    ("Cl", "Ca")    : 2.578,
    ("Cl", "Sc")    : 2.293,
    ("Cl", "Ti")    : 2.131,
    ("Cl", "V" )    : 2.103,
    ("Cl", "Cr")    : 2.146,
    ("Cl", "Mn")    : 2.207,
    ("Cl", "Fe")    : 2.126,
    ("Cl", "Co")    : 2.036,
    ("Cl", "Ni")    : 1.972,
    ("Cl", "Cu")    : 1.975,
    ("Cl", "Ga")    : 2.155,
    ("Cl", "Ge")    : 2.170,
    ("Cl", "As")    : 2.198,
    ("Cl", "Se")    : 2.167,
    ("Cl", "Br")    : 2.134,
    ("Cl", "Rb")    : 2.780,
    ("Cl", "Sr")    : 2.707,
    ("Cl", "Y" )    : 2.750,
    ("Cl", "Zr")    : 2.750,
    ("Cl", "Nb")    : 2.750,
    ("Cl", "Mo")    : 2.750,
    ("Cl", "Tc")    : 2.750,
    ("Cl", "Ru")    : 2.750,
    ("Cl", "Rh")    : 2.750,
    ("Cl", "Pd")    : 2.750,
    ("Cl", "Ag")    : 2.750,
    ("Cl", "In")    : 2.279,
    ("Cl", "Sn")    : 2.315,
    ("Cl", "Sb")    : 2.353,
    ("Cl", "Te")    : 2.367,
    ("Cl", "I" )    : 2.356,
    ("Cl", "Cs")    : 2.750,
    ("Cl", "Ba")    : 2.750,
    ("Cl", "La")    : 2.750,
    ("Cl", "Hf")    : 2.750,
    ("Cl", "Ta")    : 2.750,
    ("Cl", "W" )    : 2.750,
    ("Cl", "Re")    : 2.750,
    ("Cl", "Os")    : 2.750,
    ("Cl", "Ir")    : 2.750,
    ("Cl", "Pt")    : 2.750,
    ("Cl", "Au")    : 2.750,
    ("Cl", "Tl")    : 2.750,
    ("Cl", "Pb")    : 2.750,
    ("Cl", "Bi")    : 2.750,
    ("Cl", "Po")    : 2.750,
    ("Cl", "At")    : 2.750,
    ("K",  "K" )    : 4.240,
    ("K",  "Ca")    : 4.102,
    ("K",  "Sc")    : 2.293,
    ("K",  "Ti")    : 2.131,
    ("K",  "V" )    : 2.103,
    ("K",  "Cr")    : 2.146,
    ("K",  "Mn")    : 2.207,
    ("K",  "Fe")    : 2.126,
    ("K",  "Co")    : 2.036,
    ("K",  "Ni")    : 1.972,
    ("K",  "Cu")    : 1.975,
    ("K",  "Ga")    : 3.335,
    ("K",  "Ge")    : 3.248,
    ("K",  "As")    : 3.189,
    ("K",  "Se")    : 3.026,
    ("K",  "Br")    : 2.901,
    ("K",  "Rb")    : 3.498,
    ("K",  "Sr")    : 3.584,
    ("K",  "Y" )    : 2.750,
    ("K",  "Zr")    : 2.750,
    ("K",  "Nb")    : 2.750,
    ("K",  "Mo")    : 2.750,
    ("K",  "Tc")    : 2.750,
    ("K",  "Ru")    : 2.750,
    ("K",  "Rh")    : 2.750,
    ("K",  "Pd")    : 2.750,
    ("K",  "Ag")    : 2.750,
    ("K",  "In")    : 3.214,
    ("K",  "Sn")    : 3.195,
    ("K",  "Sb")    : 3.188,
    ("K",  "Te")    : 3.116,
    ("K",  "I" )    : 3.015,
    ("K",  "Cs")    : 2.750,
    ("K",  "Ba")    : 2.750,
    ("K",  "La")    : 2.750,
    ("K",  "Hf")    : 2.750,
    ("K",  "Ta")    : 2.750,
    ("K",  "W" )    : 2.750,
    ("K",  "Re")    : 2.750,
    ("K",  "Os")    : 2.750,
    ("K",  "Ir")    : 2.750,
    ("K",  "Pt")    : 2.750,
    ("K",  "Au")    : 2.750,
    ("K",  "Tl")    : 2.750,
    ("K",  "Pb")    : 2.750,
    ("K",  "Bi")    : 2.750,
    ("K",  "Po")    : 2.750,
    ("K",  "At")    : 2.750,
    ("Ca", "Ca")    : 3.914,
    ("Ca", "Ca", 2) : 5.739,
    ("Ca", "Sc")    : 2.293,
    ("Ca", "Ti")    : 2.131,
    ("Ca", "Ti", 2) : 2.131,
    ("Ca", "V" )    : 2.103,
    ("Ca", "V" , 2) : 2.103,
    ("Ca", "Cr")    : 2.146,
    ("Ca", "Cr", 2) : 2.146,
    ("Ca", "Mn")    : 2.207,
    ("Ca", "Mn", 2) : 2.207,
    ("Ca", "Fe")    : 2.126,
    ("Ca", "Fe", 2) : 2.126,
    ("Ca", "Co")    : 2.036,
    ("Ca", "Co", 2) : 2.036,
    ("Ca", "Ni")    : 1.972,
    ("Ca", "Ni", 2) : 1.972,
    ("Ca", "Cu")    : 1.975,
    ("Ca", "Ga")    : 3.126,
    ("Ca", "Ga", 2) : 4.464,
    ("Ca", "Ge")    : 3.036,
    ("Ca", "Ge", 2) : 3.156,
    ("Ca", "As")    : 2.970,
    ("Ca", "As", 2) : 2.775,
    ("Ca", "Se")    : 2.807,
    ("Ca", "Se", 2) : 2.589,
    ("Ca", "Br")    : 2.704,
    ("Ca", "Rb")    : 3.698,
    ("Ca", "Sr")    : 3.693,
    ("Ca", "Sr", 2) : 4.030,
    ("Ca", "Y" )    : 2.750,
    ("Ca", "Zr")    : 2.750,
    ("Ca", "Zr", 2) : 2.750,
    ("Ca", "Nb")    : 2.750,
    ("Ca", "Nb", 2) : 2.750,
    ("Ca", "Mo")    : 2.750,
    ("Ca", "Mo", 2) : 2.750,
    ("Ca", "Tc")    : 2.750,
    ("Ca", "Tc", 2) : 2.750,
    ("Ca", "Ru")    : 2.750,
    ("Ca", "Ru", 2) : 2.750,
    ("Ca", "Rh")    : 2.750,
    ("Ca", "Rh", 2) : 2.750,
    ("Ca", "Pd")    : 2.750,
    ("Ca", "Pd", 2) : 2.750,
    ("Ca", "Ag")    : 2.750,
    ("Ca", "In")    : 3.242,
    ("Ca", "In", 2) : 3.750,
    ("Ca", "Sn")    : 3.191,
    ("Ca", "Sn", 2) : 3.412,
    ("Ca", "Sb")    : 3.161,
    ("Ca", "Sb", 2) : 3.052,
    ("Ca", "Te")    : 3.044,
    ("Ca", "Te", 2) : 2.855,
    ("Ca", "I" )    : 2.934,
    ("Ca", "Cs")    : 2.750,
    ("Ca", "Ba")    : 2.750,
    ("Ca", "Ba", 2) : 2.750,
    ("Ca", "La")    : 2.750,
    ("Ca", "Hf")    : 2.750,
    ("Ca", "Hf", 2) : 2.750,
    ("Ca", "Ta")    : 2.750,
    ("Ca", "Ta", 2) : 2.750,
    ("Ca", "W" )    : 2.750,
    ("Ca", "W" , 2) : 2.750,
    ("Ca", "Re")    : 2.750,
    ("Ca", "Re", 2) : 2.750,
    ("Ca", "Os")    : 2.750,
    ("Ca", "Os", 2) : 2.750,
    ("Ca", "Ir")    : 2.750,
    ("Ca", "Ir", 2) : 2.750,
    ("Ca", "Pt")    : 2.750,
    ("Ca", "Pt", 2) : 2.750,
    ("Ca", "Au")    : 2.750,
    ("Ca", "Tl")    : 2.750,
    ("Ca", "Tl", 2) : 2.750,
    ("Ca", "Pb")    : 2.750,
    ("Ca", "Pb", 2) : 2.750,
    ("Ca", "Bi")    : 2.750,
    ("Ca", "Bi", 2) : 2.750,
    ("Ca", "Po")    : 2.750,
    ("Ca", "Po", 2) : 2.750,
    ("Ca", "At")    : 2.750,
    ("Sc", "Sc")    : 2.750,
    ("Sc", "Ti")    : 2.750,
    ("Sc", "V" )    : 2.750,
    ("Sc", "Cr")    : 2.750,
    ("Sc", "Mn")    : 2.750,
    ("Sc", "Fe")    : 2.750,
    ("Sc", "Co")    : 2.750,
    ("Sc", "Ni")    : 2.750,
    ("Sc", "Cu")    : 2.750,
    ("Sc", "Ga")    : 2.293,
    ("Sc", "Ge")    : 2.293,
    ("Sc", "As")    : 2.293,
    ("Sc", "Se")    : 2.293,
    ("Sc", "Br")    : 2.293,
    ("Sc", "Rb")    : 2.293,
    ("Sc", "Sr")    : 2.293,
    ("Sc", "Y" )    : 2.293,
    ("Sc", "Zr")    : 2.293,
    ("Sc", "Nb")    : 2.293,
    ("Sc", "Mo")    : 2.293,
    ("Sc", "Tc")    : 2.293,
    ("Sc", "Ru")    : 2.293,
    ("Sc", "Rh")    : 2.293,
    ("Sc", "Pd")    : 2.293,
    ("Sc", "Ag")    : 2.293,
    ("Sc", "In")    : 2.293,
    ("Sc", "Sn")    : 2.293,
    ("Sc", "Sb")    : 2.293,
    ("Sc", "Te")    : 2.293,
    ("Sc", "I" )    : 2.293,
    ("Sc", "Cs")    : 2.293,
    ("Sc", "Ba")    : 2.293,
    ("Sc", "La")    : 2.293,
    ("Sc", "Hf")    : 2.293,
    ("Sc", "Ta")    : 2.293,
    ("Sc", "W" )    : 2.293,
    ("Sc", "Re")    : 2.293,
    ("Sc", "Os")    : 2.293,
    ("Sc", "Ir")    : 2.293,
    ("Sc", "Pt")    : 2.293,
    ("Sc", "Au")    : 2.293,
    ("Sc", "Tl")    : 2.293,
    ("Sc", "Pb")    : 2.293,
    ("Sc", "Bi")    : 2.293,
    ("Sc", "Po")    : 2.293,
    ("Sc", "At")    : 2.293,
    ("Ti", "Ti")    : 2.750,
    ("Ti", "Ti", 2) : 2.750,
    ("Ti", "V" )    : 2.750,
    ("Ti", "V" , 2) : 2.750,
    ("Ti", "Cr")    : 2.750,
    ("Ti", "Cr", 2) : 2.750,
    ("Ti", "Mn")    : 2.750,
    ("Ti", "Mn", 2) : 2.750,
    ("Ti", "Fe")    : 2.750,
    ("Ti", "Fe", 2) : 2.750,
    ("Ti", "Co")    : 2.750,
    ("Ti", "Co", 2) : 2.750,
    ("Ti", "Ni")    : 2.750,
    ("Ti", "Ni", 2) : 2.750,
    ("Ti", "Cu")    : 2.750,
    ("Ti", "Ga")    : 2.131,
    ("Ti", "Ga", 2) : 2.131,
    ("Ti", "Ge")    : 2.131,
    ("Ti", "Ge", 2) : 2.131,
    ("Ti", "As")    : 2.131,
    ("Ti", "As", 2) : 2.131,
    ("Ti", "Se")    : 2.131,
    ("Ti", "Se", 2) : 2.131,
    ("Ti", "Br")    : 2.131,
    ("Ti", "Rb")    : 2.131,
    ("Ti", "Sr")    : 2.131,
    ("Ti", "Sr", 2) : 2.131,
    ("Ti", "Y" )    : 2.131,
    ("Ti", "Zr")    : 2.131,
    ("Ti", "Zr", 2) : 2.131,
    ("Ti", "Nb")    : 2.131,
    ("Ti", "Nb", 2) : 2.131,
    ("Ti", "Mo")    : 2.131,
    ("Ti", "Mo", 2) : 2.131,
    ("Ti", "Tc")    : 2.131,
    ("Ti", "Tc", 2) : 2.131,
    ("Ti", "Ru")    : 2.131,
    ("Ti", "Ru", 2) : 2.131,
    ("Ti", "Rh")    : 2.131,
    ("Ti", "Rh", 2) : 2.131,
    ("Ti", "Pd")    : 2.131,
    ("Ti", "Pd", 2) : 2.131,
    ("Ti", "Ag")    : 2.131,
    ("Ti", "In")    : 2.131,
    ("Ti", "In", 2) : 2.131,
    ("Ti", "Sn")    : 2.131,
    ("Ti", "Sn", 2) : 2.131,
    ("Ti", "Sb")    : 2.131,
    ("Ti", "Sb", 2) : 2.131,
    ("Ti", "Te")    : 2.131,
    ("Ti", "Te", 2) : 2.131,
    ("Ti", "I" )    : 2.131,
    ("Ti", "Cs")    : 2.131,
    ("Ti", "Ba")    : 2.131,
    ("Ti", "Ba", 2) : 2.131,
    ("Ti", "La")    : 2.131,
    ("Ti", "Hf")    : 2.131,
    ("Ti", "Hf", 2) : 2.131,
    ("Ti", "Ta")    : 2.131,
    ("Ti", "Ta", 2) : 2.131,
    ("Ti", "W" )    : 2.131,
    ("Ti", "W" , 2) : 2.131,
    ("Ti", "Re")    : 2.131,
    ("Ti", "Re", 2) : 2.131,
    ("Ti", "Os")    : 2.131,
    ("Ti", "Os", 2) : 2.131,
    ("Ti", "Ir")    : 2.131,
    ("Ti", "Ir", 2) : 2.131,
    ("Ti", "Pt")    : 2.131,
    ("Ti", "Pt", 2) : 2.131,
    ("Ti", "Au")    : 2.131,
    ("Ti", "Tl")    : 2.131,
    ("Ti", "Tl", 2) : 2.131,
    ("Ti", "Pb")    : 2.131,
    ("Ti", "Pb", 2) : 2.131,
    ("Ti", "Bi")    : 2.131,
    ("Ti", "Bi", 2) : 2.131,
    ("Ti", "Po")    : 2.131,
    ("Ti", "Po", 2) : 2.131,
    ("Ti", "At")    : 2.131,
    ("V",  "V" )    : 2.750,
    ("V",  "V" , 2) : 2.750,
    ("V",  "Cr")    : 2.750,
    ("V",  "Cr", 2) : 2.750,
    ("V",  "Mn")    : 2.750,
    ("V",  "Mn", 2) : 2.750,
    ("V",  "Fe")    : 2.750,
    ("V",  "Fe", 2) : 2.750,
    ("V",  "Co")    : 2.750,
    ("V",  "Co", 2) : 2.750,
    ("V",  "Ni")    : 2.750,
    ("V",  "Ni", 2) : 2.750,
    ("V",  "Cu")    : 2.750,
    ("V",  "Ga")    : 2.103,
    ("V",  "Ga", 2) : 2.103,
    ("V",  "Ge")    : 2.103,
    ("V",  "Ge", 2) : 2.103,
    ("V",  "As")    : 2.103,
    ("V",  "As", 2) : 2.103,
    ("V",  "Se")    : 2.103,
    ("V",  "Se", 2) : 2.103,
    ("V",  "Br")    : 2.103,
    ("V",  "Rb")    : 2.103,
    ("V",  "Sr")    : 2.103,
    ("V",  "Sr", 2) : 2.103,
    ("V",  "Y" )    : 2.103,
    ("V",  "Zr")    : 2.103,
    ("V",  "Zr", 2) : 2.103,
    ("V",  "Nb")    : 2.103,
    ("V",  "Nb", 2) : 2.103,
    ("V",  "Mo")    : 2.103,
    ("V",  "Mo", 2) : 2.103,
    ("V",  "Tc")    : 2.103,
    ("V",  "Tc", 2) : 2.103,
    ("V",  "Ru")    : 2.103,
    ("V",  "Ru", 2) : 2.103,
    ("V",  "Rh")    : 2.103,
    ("V",  "Rh", 2) : 2.103,
    ("V",  "Pd")    : 2.103,
    ("V",  "Pd", 2) : 2.103,
    ("V",  "Ag")    : 2.103,
    ("V",  "In")    : 2.103,
    ("V",  "In", 2) : 2.103,
    ("V",  "Sn")    : 2.103,
    ("V",  "Sn", 2) : 2.103,
    ("V",  "Sb")    : 2.103,
    ("V",  "Sb", 2) : 2.103,
    ("V",  "Te")    : 2.103,
    ("V",  "Te", 2) : 2.103,
    ("V",  "I" )    : 2.103,
    ("V",  "Cs")    : 2.103,
    ("V",  "Ba")    : 2.103,
    ("V",  "Ba", 2) : 2.103,
    ("V",  "La")    : 2.103,
    ("V",  "Hf")    : 2.103,
    ("V",  "Hf", 2) : 2.103,
    ("V",  "Ta")    : 2.103,
    ("V",  "Ta", 2) : 2.103,
    ("V",  "W" )    : 2.103,
    ("V",  "W" , 2) : 2.103,
    ("V",  "Re")    : 2.103,
    ("V",  "Re", 2) : 2.103,
    ("V",  "Os")    : 2.103,
    ("V",  "Os", 2) : 2.103,
    ("V",  "Ir")    : 2.103,
    ("V",  "Ir", 2) : 2.103,
    ("V",  "Pt")    : 2.103,
    ("V",  "Pt", 2) : 2.103,
    ("V",  "Au")    : 2.103,
    ("V",  "Tl")    : 2.103,
    ("V",  "Tl", 2) : 2.103,
    ("V",  "Pb")    : 2.103,
    ("V",  "Pb", 2) : 2.103,
    ("V",  "Bi")    : 2.103,
    ("V",  "Bi", 2) : 2.103,
    ("V",  "Po")    : 2.103,
    ("V",  "Po", 2) : 2.103,
    ("V",  "At")    : 2.103,
    ("Cr", "Cr")    : 2.750,
    ("Cr", "Cr", 2) : 2.750,
    ("Cr", "Mn")    : 2.750,
    ("Cr", "Mn", 2) : 2.750,
    ("Cr", "Fe")    : 2.750,
    ("Cr", "Fe", 2) : 2.750,
    ("Cr", "Co")    : 2.750,
    ("Cr", "Co", 2) : 2.750,
    ("Cr", "Ni")    : 2.750,
    ("Cr", "Ni", 2) : 2.750,
    ("Cr", "Cu")    : 2.750,
    ("Cr", "Ga")    : 2.146,
    ("Cr", "Ga", 2) : 2.146,
    ("Cr", "Ge")    : 2.146,
    ("Cr", "Ge", 2) : 2.146,
    ("Cr", "As")    : 2.146,
    ("Cr", "As", 2) : 2.146,
    ("Cr", "Se")    : 2.146,
    ("Cr", "Se", 2) : 2.146,
    ("Cr", "Br")    : 2.146,
    ("Cr", "Rb")    : 2.146,
    ("Cr", "Sr")    : 2.146,
    ("Cr", "Sr", 2) : 2.146,
    ("Cr", "Y" )    : 2.146,
    ("Cr", "Zr")    : 2.146,
    ("Cr", "Zr", 2) : 2.146,
    ("Cr", "Nb")    : 2.146,
    ("Cr", "Nb", 2) : 2.146,
    ("Cr", "Mo")    : 2.146,
    ("Cr", "Mo", 2) : 2.146,
    ("Cr", "Tc")    : 2.146,
    ("Cr", "Tc", 2) : 2.146,
    ("Cr", "Ru")    : 2.146,
    ("Cr", "Ru", 2) : 2.146,
    ("Cr", "Rh")    : 2.146,
    ("Cr", "Rh", 2) : 2.146,
    ("Cr", "Pd")    : 2.146,
    ("Cr", "Pd", 2) : 2.146,
    ("Cr", "Ag")    : 2.146,
    ("Cr", "In")    : 2.146,
    ("Cr", "In", 2) : 2.146,
    ("Cr", "Sn")    : 2.146,
    ("Cr", "Sn", 2) : 2.146,
    ("Cr", "Sb")    : 2.146,
    ("Cr", "Sb", 2) : 2.146,
    ("Cr", "Te")    : 2.146,
    ("Cr", "Te", 2) : 2.146,
    ("Cr", "I" )    : 2.146,
    ("Cr", "Cs")    : 2.146,
    ("Cr", "Ba")    : 2.146,
    ("Cr", "Ba", 2) : 2.146,
    ("Cr", "La")    : 2.146,
    ("Cr", "Hf")    : 2.146,
    ("Cr", "Hf", 2) : 2.146,
    ("Cr", "Ta")    : 2.146,
    ("Cr", "Ta", 2) : 2.146,
    ("Cr", "W" )    : 2.146,
    ("Cr", "W" , 2) : 2.146,
    ("Cr", "Re")    : 2.146,
    ("Cr", "Re", 2) : 2.146,
    ("Cr", "Os")    : 2.146,
    ("Cr", "Os", 2) : 2.146,
    ("Cr", "Ir")    : 2.146,
    ("Cr", "Ir", 2) : 2.146,
    ("Cr", "Pt")    : 2.146,
    ("Cr", "Pt", 2) : 2.146,
    ("Cr", "Au")    : 2.146,
    ("Cr", "Tl")    : 2.146,
    ("Cr", "Tl", 2) : 2.146,
    ("Cr", "Pb")    : 2.146,
    ("Cr", "Pb", 2) : 2.146,
    ("Cr", "Bi")    : 2.146,
    ("Cr", "Bi", 2) : 2.146,
    ("Cr", "Po")    : 2.146,
    ("Cr", "Po", 2) : 2.146,
    ("Cr", "At")    : 2.146,
    ("Mn", "Mn")    : 2.750,
    ("Mn", "Mn", 2) : 2.750,
    ("Mn", "Fe")    : 2.750,
    ("Mn", "Fe", 2) : 2.750,
    ("Mn", "Co")    : 2.750,
    ("Mn", "Co", 2) : 2.750,
    ("Mn", "Ni")    : 2.750,
    ("Mn", "Ni", 2) : 2.750,
    ("Mn", "Cu")    : 2.750,
    ("Mn", "Ga")    : 2.207,
    ("Mn", "Ga", 2) : 2.207,
    ("Mn", "Ge")    : 2.207,
    ("Mn", "Ge", 2) : 2.207,
    ("Mn", "As")    : 2.207,
    ("Mn", "As", 2) : 2.207,
    ("Mn", "Se")    : 2.207,
    ("Mn", "Se", 2) : 2.207,
    ("Mn", "Br")    : 2.207,
    ("Mn", "Rb")    : 2.207,
    ("Mn", "Sr")    : 2.207,
    ("Mn", "Sr", 2) : 2.207,
    ("Mn", "Y" )    : 2.207,
    ("Mn", "Zr")    : 2.207,
    ("Mn", "Zr", 2) : 2.207,
    ("Mn", "Nb")    : 2.207,
    ("Mn", "Nb", 2) : 2.207,
    ("Mn", "Mo")    : 2.207,
    ("Mn", "Mo", 2) : 2.207,
    ("Mn", "Tc")    : 2.207,
    ("Mn", "Tc", 2) : 2.207,
    ("Mn", "Ru")    : 2.207,
    ("Mn", "Ru", 2) : 2.207,
    ("Mn", "Rh")    : 2.207,
    ("Mn", "Rh", 2) : 2.207,
    ("Mn", "Pd")    : 2.207,
    ("Mn", "Pd", 2) : 2.207,
    ("Mn", "Ag")    : 2.207,
    ("Mn", "In")    : 2.207,
    ("Mn", "In", 2) : 2.207,
    ("Mn", "Sn")    : 2.207,
    ("Mn", "Sn", 2) : 2.207,
    ("Mn", "Sb")    : 2.207,
    ("Mn", "Sb", 2) : 2.207,
    ("Mn", "Te")    : 2.207,
    ("Mn", "Te", 2) : 2.207,
    ("Mn", "I" )    : 2.207,
    ("Mn", "Cs")    : 2.207,
    ("Mn", "Ba")    : 2.207,
    ("Mn", "Ba", 2) : 2.207,
    ("Mn", "La")    : 2.207,
    ("Mn", "Hf")    : 2.207,
    ("Mn", "Hf", 2) : 2.207,
    ("Mn", "Ta")    : 2.207,
    ("Mn", "Ta", 2) : 2.207,
    ("Mn", "W" )    : 2.207,
    ("Mn", "W" , 2) : 2.207,
    ("Mn", "Re")    : 2.207,
    ("Mn", "Re", 2) : 2.207,
    ("Mn", "Os")    : 2.207,
    ("Mn", "Os", 2) : 2.207,
    ("Mn", "Ir")    : 2.207,
    ("Mn", "Ir", 2) : 2.207,
    ("Mn", "Pt")    : 2.207,
    ("Mn", "Pt", 2) : 2.207,
    ("Mn", "Au")    : 2.207,
    ("Mn", "Tl")    : 2.207,
    ("Mn", "Tl", 2) : 2.207,
    ("Mn", "Pb")    : 2.207,
    ("Mn", "Pb", 2) : 2.207,
    ("Mn", "Bi")    : 2.207,
    ("Mn", "Bi", 2) : 2.207,
    ("Mn", "Po")    : 2.207,
    ("Mn", "Po", 2) : 2.207,
    ("Mn", "At")    : 2.207,
    ("Fe", "Fe")    : 2.750,
    ("Fe", "Fe", 2) : 2.750,
    ("Fe", "Co")    : 2.750,
    ("Fe", "Co", 2) : 2.750,
    ("Fe", "Ni")    : 2.750,
    ("Fe", "Ni", 2) : 2.750,
    ("Fe", "Cu")    : 2.750,
    ("Fe", "Ga")    : 2.126,
    ("Fe", "Ga", 2) : 2.126,
    ("Fe", "Ge")    : 2.126,
    ("Fe", "Ge", 2) : 2.126,
    ("Fe", "As")    : 2.126,
    ("Fe", "As", 2) : 2.126,
    ("Fe", "Se")    : 2.126,
    ("Fe", "Se", 2) : 2.126,
    ("Fe", "Br")    : 2.126,
    ("Fe", "Rb")    : 2.126,
    ("Fe", "Sr")    : 2.126,
    ("Fe", "Sr", 2) : 2.126,
    ("Fe", "Y" )    : 2.126,
    ("Fe", "Zr")    : 2.126,
    ("Fe", "Zr", 2) : 2.126,
    ("Fe", "Nb")    : 2.126,
    ("Fe", "Nb", 2) : 2.126,
    ("Fe", "Mo")    : 2.126,
    ("Fe", "Mo", 2) : 2.126,
    ("Fe", "Tc")    : 2.126,
    ("Fe", "Tc", 2) : 2.126,
    ("Fe", "Ru")    : 2.126,
    ("Fe", "Ru", 2) : 2.126,
    ("Fe", "Rh")    : 2.126,
    ("Fe", "Rh", 2) : 2.126,
    ("Fe", "Pd")    : 2.126,
    ("Fe", "Pd", 2) : 2.126,
    ("Fe", "Ag")    : 2.126,
    ("Fe", "In")    : 2.126,
    ("Fe", "In", 2) : 2.126,
    ("Fe", "Sn")    : 2.126,
    ("Fe", "Sn", 2) : 2.126,
    ("Fe", "Sb")    : 2.126,
    ("Fe", "Sb", 2) : 2.126,
    ("Fe", "Te")    : 2.126,
    ("Fe", "Te", 2) : 2.126,
    ("Fe", "I" )    : 2.126,
    ("Fe", "Cs")    : 2.126,
    ("Fe", "Ba")    : 2.126,
    ("Fe", "Ba", 2) : 2.126,
    ("Fe", "La")    : 2.126,
    ("Fe", "Hf")    : 2.126,
    ("Fe", "Hf", 2) : 2.126,
    ("Fe", "Ta")    : 2.126,
    ("Fe", "Ta", 2) : 2.126,
    ("Fe", "W" )    : 2.126,
    ("Fe", "W" , 2) : 2.126,
    ("Fe", "Re")    : 2.126,
    ("Fe", "Re", 2) : 2.126,
    ("Fe", "Os")    : 2.126,
    ("Fe", "Os", 2) : 2.126,
    ("Fe", "Ir")    : 2.126,
    ("Fe", "Ir", 2) : 2.126,
    ("Fe", "Pt")    : 2.126,
    ("Fe", "Pt", 2) : 2.126,
    ("Fe", "Au")    : 2.126,
    ("Fe", "Tl")    : 2.126,
    ("Fe", "Tl", 2) : 2.126,
    ("Fe", "Pb")    : 2.126,
    ("Fe", "Pb", 2) : 2.126,
    ("Fe", "Bi")    : 2.126,
    ("Fe", "Bi", 2) : 2.126,
    ("Fe", "Po")    : 2.126,
    ("Fe", "Po", 2) : 2.126,
    ("Fe", "At")    : 2.126,
    ("Co", "Co")    : 2.750,
    ("Co", "Co", 2) : 2.750,
    ("Co", "Ni")    : 2.750,
    ("Co", "Ni", 2) : 2.750,
    ("Co", "Cu")    : 2.750,
    ("Co", "Ga")    : 2.036,
    ("Co", "Ga", 2) : 2.036,
    ("Co", "Ge")    : 2.036,
    ("Co", "Ge", 2) : 2.036,
    ("Co", "As")    : 2.036,
    ("Co", "As", 2) : 2.036,
    ("Co", "Se")    : 2.036,
    ("Co", "Se", 2) : 2.036,
    ("Co", "Br")    : 2.036,
    ("Co", "Rb")    : 2.036,
    ("Co", "Sr")    : 2.036,
    ("Co", "Sr", 2) : 2.036,
    ("Co", "Y" )    : 2.036,
    ("Co", "Zr")    : 2.036,
    ("Co", "Zr", 2) : 2.036,
    ("Co", "Nb")    : 2.036,
    ("Co", "Nb", 2) : 2.036,
    ("Co", "Mo")    : 2.036,
    ("Co", "Mo", 2) : 2.036,
    ("Co", "Tc")    : 2.036,
    ("Co", "Tc", 2) : 2.036,
    ("Co", "Ru")    : 2.036,
    ("Co", "Ru", 2) : 2.036,
    ("Co", "Rh")    : 2.036,
    ("Co", "Rh", 2) : 2.036,
    ("Co", "Pd")    : 2.036,
    ("Co", "Pd", 2) : 2.036,
    ("Co", "Ag")    : 2.036,
    ("Co", "In")    : 2.036,
    ("Co", "In", 2) : 2.036,
    ("Co", "Sn")    : 2.036,
    ("Co", "Sn", 2) : 2.036,
    ("Co", "Sb")    : 2.036,
    ("Co", "Sb", 2) : 2.036,
    ("Co", "Te")    : 2.036,
    ("Co", "Te", 2) : 2.036,
    ("Co", "I" )    : 2.036,
    ("Co", "Cs")    : 2.036,
    ("Co", "Ba")    : 2.036,
    ("Co", "Ba", 2) : 2.036,
    ("Co", "La")    : 2.036,
    ("Co", "Hf")    : 2.036,
    ("Co", "Hf", 2) : 2.036,
    ("Co", "Ta")    : 2.036,
    ("Co", "Ta", 2) : 2.036,
    ("Co", "W" )    : 2.036,
    ("Co", "W" , 2) : 2.036,
    ("Co", "Re")    : 2.036,
    ("Co", "Re", 2) : 2.036,
    ("Co", "Os")    : 2.036,
    ("Co", "Os", 2) : 2.036,
    ("Co", "Ir")    : 2.036,
    ("Co", "Ir", 2) : 2.036,
    ("Co", "Pt")    : 2.036,
    ("Co", "Pt", 2) : 2.036,
    ("Co", "Au")    : 2.036,
    ("Co", "Tl")    : 2.036,
    ("Co", "Tl", 2) : 2.036,
    ("Co", "Pb")    : 2.036,
    ("Co", "Pb", 2) : 2.036,
    ("Co", "Bi")    : 2.036,
    ("Co", "Bi", 2) : 2.036,
    ("Co", "Po")    : 2.036,
    ("Co", "Po", 2) : 2.036,
    ("Co", "At")    : 2.036,
    ("Ni", "Ni")    : 2.750,
    ("Ni", "Ni", 2) : 2.750,
    ("Ni", "Cu")    : 2.750,
    ("Ni", "Ga")    : 1.972,
    ("Ni", "Ga", 2) : 1.972,
    ("Ni", "Ge")    : 1.972,
    ("Ni", "Ge", 2) : 1.972,
    ("Ni", "As")    : 1.972,
    ("Ni", "As", 2) : 1.972,
    ("Ni", "Se")    : 1.972,
    ("Ni", "Se", 2) : 1.972,
    ("Ni", "Br")    : 1.972,
    ("Ni", "Rb")    : 1.972,
    ("Ni", "Sr")    : 1.972,
    ("Ni", "Sr", 2) : 1.972,
    ("Ni", "Y" )    : 1.972,
    ("Ni", "Zr")    : 1.972,
    ("Ni", "Zr", 2) : 1.972,
    ("Ni", "Nb")    : 1.972,
    ("Ni", "Nb", 2) : 1.972,
    ("Ni", "Mo")    : 1.972,
    ("Ni", "Mo", 2) : 1.972,
    ("Ni", "Tc")    : 1.972,
    ("Ni", "Tc", 2) : 1.972,
    ("Ni", "Ru")    : 1.972,
    ("Ni", "Ru", 2) : 1.972,
    ("Ni", "Rh")    : 1.972,
    ("Ni", "Rh", 2) : 1.972,
    ("Ni", "Pd")    : 1.972,
    ("Ni", "Pd", 2) : 1.972,
    ("Ni", "Ag")    : 1.972,
    ("Ni", "In")    : 1.972,
    ("Ni", "In", 2) : 1.972,
    ("Ni", "Sn")    : 1.972,
    ("Ni", "Sn", 2) : 1.972,
    ("Ni", "Sb")    : 1.972,
    ("Ni", "Sb", 2) : 1.972,
    ("Ni", "Te")    : 1.972,
    ("Ni", "Te", 2) : 1.972,
    ("Ni", "I" )    : 1.972,
    ("Ni", "Cs")    : 1.972,
    ("Ni", "Ba")    : 1.972,
    ("Ni", "Ba", 2) : 1.972,
    ("Ni", "La")    : 1.972,
    ("Ni", "Hf")    : 1.972,
    ("Ni", "Hf", 2) : 1.972,
    ("Ni", "Ta")    : 1.972,
    ("Ni", "Ta", 2) : 1.972,
    ("Ni", "W" )    : 1.972,
    ("Ni", "W" , 2) : 1.972,
    ("Ni", "Re")    : 1.972,
    ("Ni", "Re", 2) : 1.972,
    ("Ni", "Os")    : 1.972,
    ("Ni", "Os", 2) : 1.972,
    ("Ni", "Ir")    : 1.972,
    ("Ni", "Ir", 2) : 1.972,
    ("Ni", "Pt")    : 1.972,
    ("Ni", "Pt", 2) : 1.972,
    ("Ni", "Au")    : 1.972,
    ("Ni", "Tl")    : 1.972,
    ("Ni", "Tl", 2) : 1.972,
    ("Ni", "Pb")    : 1.972,
    ("Ni", "Pb", 2) : 1.972,
    ("Ni", "Bi")    : 1.972,
    ("Ni", "Bi", 2) : 1.972,
    ("Ni", "Po")    : 1.972,
    ("Ni", "Po", 2) : 1.972,
    ("Ni", "At")    : 1.972,
    ("Cu", "Cu")    : 2.750,
    ("Cu", "Ga")    : 1.975,
    ("Cu", "Ge")    : 1.975,
    ("Cu", "As")    : 1.975,
    ("Cu", "Se")    : 1.975,
    ("Cu", "Br")    : 1.975,
    ("Cu", "Rb")    : 1.975,
    ("Cu", "Sr")    : 1.975,
    ("Cu", "Y" )    : 1.975,
    ("Cu", "Zr")    : 1.975,
    ("Cu", "Nb")    : 1.975,
    ("Cu", "Mo")    : 1.975,
    ("Cu", "Tc")    : 1.975,
    ("Cu", "Ru")    : 1.975,
    ("Cu", "Rh")    : 1.975,
    ("Cu", "Pd")    : 1.975,
    ("Cu", "Ag")    : 1.975,
    ("Cu", "In")    : 1.975,
    ("Cu", "Sn")    : 1.975,
    ("Cu", "Sb")    : 1.975,
    ("Cu", "Te")    : 1.975,
    ("Cu", "I" )    : 1.975,
    ("Cu", "Cs")    : 1.975,
    ("Cu", "Ba")    : 1.975,
    ("Cu", "La")    : 1.975,
    ("Cu", "Hf")    : 1.975,
    ("Cu", "Ta")    : 1.975,
    ("Cu", "W" )    : 1.975,
    ("Cu", "Re")    : 1.975,
    ("Cu", "Os")    : 1.975,
    ("Cu", "Ir")    : 1.975,
    ("Cu", "Pt")    : 1.975,
    ("Cu", "Au")    : 1.975,
    ("Cu", "Tl")    : 1.975,
    ("Cu", "Pb")    : 1.975,
    ("Cu", "Bi")    : 1.975,
    ("Cu", "Po")    : 1.975,
    ("Cu", "At")    : 1.975,
    ("Ga", "Ga")    : 2.471,
    ("Ga", "Ga", 2) : 2.578,
    ("Ga", "Ge")    : 2.440,
    ("Ga", "Ge", 2) : 2.394,
    ("Ga", "As")    : 2.430,
    ("Ga", "As", 2) : 2.204,
    ("Ga", "Se")    : 2.306,
    ("Ga", "Se", 2) : 2.102,
    ("Ga", "Br")    : 2.284,
    ("Ga", "Rb")    : 2.750,
    ("Ga", "Sr")    : 2.750,
    ("Ga", "Sr", 2) : 3.287,
    ("Ga", "Y" )    : 2.750,
    ("Ga", "Zr")    : 2.750,
    ("Ga", "Zr", 2) : 2.750,
    ("Ga", "Nb")    : 2.750,
    ("Ga", "Nb", 2) : 2.750,
    ("Ga", "Mo")    : 2.750,
    ("Ga", "Mo", 2) : 2.750,
    ("Ga", "Tc")    : 2.750,
    ("Ga", "Tc", 2) : 2.750,
    ("Ga", "Ru")    : 2.750,
    ("Ga", "Ru", 2) : 2.750,
    ("Ga", "Rh")    : 2.750,
    ("Ga", "Rh", 2) : 2.750,
    ("Ga", "Pd")    : 2.750,
    ("Ga", "Pd", 2) : 2.750,
    ("Ga", "Ag")    : 2.750,
    ("Ga", "In")    : 2.345,
    ("Ga", "In", 2) : 2.147,
    ("Ga", "Sn")    : 2.577,
    ("Ga", "Sn", 2) : 2.187,
    ("Ga", "Sb")    : 2.634,
    ("Ga", "Sb", 2) : 2.231,
    ("Ga", "Te")    : 2.459,
    ("Ga", "Te", 2) : 2.172,
    ("Ga", "I" )    : 2.469,
    ("Ga", "Cs")    : 2.750,
    ("Ga", "Ba")    : 2.750,
    ("Ga", "Ba", 2) : 2.750,
    ("Ga", "La")    : 2.750,
    ("Ga", "Hf")    : 2.750,
    ("Ga", "Hf", 2) : 2.750,
    ("Ga", "Ta")    : 2.750,
    ("Ga", "Ta", 2) : 2.750,
    ("Ga", "W" )    : 2.750,
    ("Ga", "W" , 2) : 2.750,
    ("Ga", "Re")    : 2.750,
    ("Ga", "Re", 2) : 2.750,
    ("Ga", "Os")    : 2.750,
    ("Ga", "Os", 2) : 2.750,
    ("Ga", "Ir")    : 2.750,
    ("Ga", "Ir", 2) : 2.750,
    ("Ga", "Pt")    : 2.750,
    ("Ga", "Pt", 2) : 2.750,
    ("Ga", "Au")    : 2.750,
    ("Ga", "Tl")    : 2.750,
    ("Ga", "Tl", 2) : 2.750,
    ("Ga", "Pb")    : 2.750,
    ("Ga", "Pb", 2) : 2.750,
    ("Ga", "Bi")    : 2.750,
    ("Ga", "Bi", 2) : 2.750,
    ("Ga", "Po")    : 2.750,
    ("Ga", "Po", 2) : 2.750,
    ("Ga", "At")    : 2.750,
    ("Ge", "Ge")    : 2.413,
    ("Ge", "Ge", 2) : 2.215,
    ("Ge", "As")    : 2.407,
    ("Ge", "As", 2) : 2.194,
    ("Ge", "Se")    : 2.347,
    ("Ge", "Se", 2) : 2.123,
    ("Ge", "Br")    : 2.306,
    ("Ge", "Rb")    : 3.114,
    ("Ge", "Sr")    : 3.076,
    ("Ge", "Sr", 2) : 3.137,
    ("Ge", "Y" )    : 2.750,
    ("Ge", "Zr")    : 2.750,
    ("Ge", "Zr", 2) : 2.750,
    ("Ge", "Nb")    : 2.750,
    ("Ge", "Nb", 2) : 2.750,
    ("Ge", "Mo")    : 2.750,
    ("Ge", "Mo", 2) : 2.750,
    ("Ge", "Tc")    : 2.750,
    ("Ge", "Tc", 2) : 2.750,
    ("Ge", "Ru")    : 2.750,
    ("Ge", "Ru", 2) : 2.750,
    ("Ge", "Rh")    : 2.750,
    ("Ge", "Rh", 2) : 2.750,
    ("Ge", "Pd")    : 2.750,
    ("Ge", "Pd", 2) : 2.750,
    ("Ge", "Ag")    : 2.750,
    ("Ge", "In")    : 2.541,
    ("Ge", "In", 2) : 2.183,
    ("Ge", "Sn")    : 2.566,
    ("Ge", "Sn", 2) : 2.223,
    ("Ge", "Sb")    : 2.583,
    ("Ge", "Sb", 2) : 2.308,
    ("Ge", "Te")    : 2.548,
    ("Ge", "Te", 2) : 2.292,
    ("Ge", "I" )    : 2.507,
    ("Ge", "Cs")    : 2.750,
    ("Ge", "Ba")    : 2.750,
    ("Ge", "Ba", 2) : 2.750,
    ("Ge", "La")    : 2.750,
    ("Ge", "Hf")    : 2.750,
    ("Ge", "Hf", 2) : 2.750,
    ("Ge", "Ta")    : 2.750,
    ("Ge", "Ta", 2) : 2.750,
    ("Ge", "W" )    : 2.750,
    ("Ge", "W" , 2) : 2.750,
    ("Ge", "Re")    : 2.750,
    ("Ge", "Re", 2) : 2.750,
    ("Ge", "Os")    : 2.750,
    ("Ge", "Os", 2) : 2.750,
    ("Ge", "Ir")    : 2.750,
    ("Ge", "Ir", 2) : 2.750,
    ("Ge", "Pt")    : 2.750,
    ("Ge", "Pt", 2) : 2.750,
    ("Ge", "Au")    : 2.750,
    ("Ge", "Tl")    : 2.750,
    ("Ge", "Tl", 2) : 2.750,
    ("Ge", "Pb")    : 2.750,
    ("Ge", "Pb", 2) : 2.750,
    ("Ge", "Bi")    : 2.750,
    ("Ge", "Bi", 2) : 2.750,
    ("Ge", "Po")    : 2.750,
    ("Ge", "Po", 2) : 2.750,
    ("Ge", "At")    : 2.750,
    ("As", "As")    : 2.419,
    ("As", "As", 2) : 2.203,
    ("As", "Se")    : 2.358,
    ("As", "Se", 2) : 2.145,
    ("As", "Br")    : 2.335,
    ("As", "Rb")    : 3.062,
    ("As", "Sr")    : 3.032,
    ("As", "Sr", 2) : 2.957,
    ("As", "Y" )    : 2.750,
    ("As", "Zr")    : 2.750,
    ("As", "Zr", 2) : 2.750,
    ("As", "Nb")    : 2.750,
    ("As", "Nb", 2) : 2.750,
    ("As", "Mo")    : 2.750,
    ("As", "Mo", 2) : 2.750,
    ("As", "Tc")    : 2.750,
    ("As", "Tc", 2) : 2.750,
    ("As", "Ru")    : 2.750,
    ("As", "Ru", 2) : 2.750,
    ("As", "Rh")    : 2.750,
    ("As", "Rh", 2) : 2.750,
    ("As", "Pd")    : 2.750,
    ("As", "Pd", 2) : 2.750,
    ("As", "Ag")    : 2.750,
    ("As", "In")    : 2.566,
    ("As", "In", 2) : 2.216,
    ("As", "Sn")    : 2.567,
    ("As", "Sn", 2) : 2.304,
    ("As", "Sb")    : 2.592,
    ("As", "Sb", 2) : 2.364,
    ("As", "Te")    : 2.567,
    ("As", "Te", 2) : 2.356,
    ("As", "I" )    : 2.544,
    ("As", "Cs")    : 2.750,
    ("As", "Ba")    : 2.750,
    ("As", "Ba", 2) : 2.750,
    ("As", "La")    : 2.750,
    ("As", "Hf")    : 2.750,
    ("As", "Hf", 2) : 2.750,
    ("As", "Ta")    : 2.750,
    ("As", "Ta", 2) : 2.750,
    ("As", "W" )    : 2.750,
    ("As", "W" , 2) : 2.750,
    ("As", "Re")    : 2.750,
    ("As", "Re", 2) : 2.750,
    ("As", "Os")    : 2.750,
    ("As", "Os", 2) : 2.750,
    ("As", "Ir")    : 2.750,
    ("As", "Ir", 2) : 2.750,
    ("As", "Pt")    : 2.750,
    ("As", "Pt", 2) : 2.750,
    ("As", "Au")    : 2.750,
    ("As", "Tl")    : 2.750,
    ("As", "Tl", 2) : 2.750,
    ("As", "Pb")    : 2.750,
    ("As", "Pb", 2) : 2.750,
    ("As", "Bi")    : 2.750,
    ("As", "Bi", 2) : 2.750,
    ("As", "Po")    : 2.750,
    ("As", "Po", 2) : 2.750,
    ("As", "At")    : 2.750,
    ("Se", "Se")    : 2.312,
    ("Se", "Se", 2) : 2.135,
    ("Se", "Br")    : 2.305,
    ("Se", "Rb")    : 3.000,
    ("Se", "Sr")    : 2.943,
    ("Se", "Sr", 2) : 2.804,
    ("Se", "Y" )    : 2.750,
    ("Se", "Zr")    : 2.750,
    ("Se", "Zr", 2) : 2.750,
    ("Se", "Nb")    : 2.750,
    ("Se", "Nb", 2) : 2.750,
    ("Se", "Mo")    : 2.750,
    ("Se", "Mo", 2) : 2.750,
    ("Se", "Tc")    : 2.750,
    ("Se", "Tc", 2) : 2.750,
    ("Se", "Ru")    : 2.750,
    ("Se", "Ru", 2) : 2.750,
    ("Se", "Rh")    : 2.750,
    ("Se", "Rh", 2) : 2.750,
    ("Se", "Pd")    : 2.750,
    ("Se", "Pd", 2) : 2.750,
    ("Se", "Ag")    : 2.750,
    ("Se", "In")    : 2.505,
    ("Se", "In", 2) : 2.153,
    ("Se", "Sn")    : 2.510,
    ("Se", "Sn", 2) : 2.261,
    ("Se", "Sb")    : 2.536,
    ("Se", "Sb", 2) : 2.323,
    ("Se", "Te")    : 2.560,
    ("Se", "Te", 2) : 2.358,
    ("Se", "I" )    : 2.524,
    ("Se", "Cs")    : 2.750,
    ("Se", "Ba")    : 2.750,
    ("Se", "Ba", 2) : 2.750,
    ("Se", "La")    : 2.750,
    ("Se", "Hf")    : 2.750,
    ("Se", "Hf", 2) : 2.750,
    ("Se", "Ta")    : 2.750,
    ("Se", "Ta", 2) : 2.750,
    ("Se", "W" )    : 2.750,
    ("Se", "W" , 2) : 2.750,
    ("Se", "Re")    : 2.750,
    ("Se", "Re", 2) : 2.750,
    ("Se", "Os")    : 2.750,
    ("Se", "Os", 2) : 2.750,
    ("Se", "Ir")    : 2.750,
    ("Se", "Ir", 2) : 2.750,
    ("Se", "Pt")    : 2.750,
    ("Se", "Pt", 2) : 2.750,
    ("Se", "Au")    : 2.750,
    ("Se", "Tl")    : 2.750,
    ("Se", "Tl", 2) : 2.750,
    ("Se", "Pb")    : 2.750,
    ("Se", "Pb", 2) : 2.750,
    ("Se", "Bi")    : 2.750,
    ("Se", "Bi", 2) : 2.750,
    ("Se", "Po")    : 2.750,
    ("Se", "Po", 2) : 2.750,
    ("Se", "At")    : 2.750,
    ("Br", "Br")    : 2.274,
    ("Br", "Rb")    : 2.908,
    ("Br", "Sr")    : 2.843,
    ("Br", "Y" )    : 2.750,
    ("Br", "Zr")    : 2.750,
    ("Br", "Nb")    : 2.750,
    ("Br", "Mo")    : 2.750,
    ("Br", "Tc")    : 2.750,
    ("Br", "Ru")    : 2.750,
    ("Br", "Rh")    : 2.750,
    ("Br", "Pd")    : 2.750,
    ("Br", "Ag")    : 2.750,
    ("Br", "In")    : 2.402,
    ("Br", "Sn")    : 2.448,
    ("Br", "Sb")    : 2.491,
    ("Br", "Te")    : 2.501,
    ("Br", "I" )    : 2.489,
    ("Br", "Cs")    : 2.750,
    ("Br", "Ba")    : 2.750,
    ("Br", "La")    : 2.750,
    ("Br", "Hf")    : 2.750,
    ("Br", "Ta")    : 2.750,
    ("Br", "W" )    : 2.750,
    ("Br", "Re")    : 2.750,
    ("Br", "Os")    : 2.750,
    ("Br", "Ir")    : 2.750,
    ("Br", "Pt")    : 2.750,
    ("Br", "Au")    : 2.750,
    ("Br", "Tl")    : 2.750,
    ("Br", "Pb")    : 2.750,
    ("Br", "Bi")    : 2.750,
    ("Br", "Po")    : 2.750,
    ("Br", "At")    : 2.750,
    ("Rb", "Rb")    : 3.510,
    ("Rb", "Sr")    : 3.642,
    ("Rb", "Y" )    : 2.750,
    ("Rb", "Zr")    : 2.750,
    ("Rb", "Nb")    : 2.750,
    ("Rb", "Mo")    : 2.750,
    ("Rb", "Tc")    : 2.750,
    ("Rb", "Ru")    : 2.750,
    ("Rb", "Rh")    : 2.750,
    ("Rb", "Pd")    : 2.750,
    ("Rb", "Ag")    : 2.750,
    ("Rb", "In")    : 3.315,
    ("Rb", "Sn")    : 3.294,
    ("Rb", "Sb")    : 3.300,
    ("Rb", "Te")    : 3.247,
    ("Rb", "I" )    : 3.155,
    ("Rb", "Cs")    : 2.750,
    ("Rb", "Ba")    : 2.750,
    ("Rb", "La")    : 2.750,
    ("Rb", "Hf")    : 2.750,
    ("Rb", "Ta")    : 2.750,
    ("Rb", "W" )    : 2.750,
    ("Rb", "Re")    : 2.750,
    ("Rb", "Os")    : 2.750,
    ("Rb", "Ir")    : 2.750,
    ("Rb", "Pt")    : 2.750,
    ("Rb", "Au")    : 2.750,
    ("Rb", "Tl")    : 2.750,
    ("Rb", "Pb")    : 2.750,
    ("Rb", "Bi")    : 2.750,
    ("Rb", "Po")    : 2.750,
    ("Rb", "At")    : 2.750,
    ("Sr", "Sr")    : 3.705,
    ("Sr", "Sr", 2) : 4.178,
    ("Sr", "Y" )    : 2.750,
    ("Sr", "Zr")    : 2.750,
    ("Sr", "Zr", 2) : 2.750,
    ("Sr", "Nb")    : 2.750,
    ("Sr", "Nb", 2) : 2.750,
    ("Sr", "Mo")    : 2.750,
    ("Sr", "Mo", 2) : 2.750,
    ("Sr", "Tc")    : 2.750,
    ("Sr", "Tc", 2) : 2.750,
    ("Sr", "Ru")    : 2.750,
    ("Sr", "Ru", 2) : 2.750,
    ("Sr", "Rh")    : 2.750,
    ("Sr", "Rh", 2) : 2.750,
    ("Sr", "Pd")    : 2.750,
    ("Sr", "Pd", 2) : 2.750,
    ("Sr", "Ag")    : 2.750,
    ("Sr", "In")    : 3.314,
    ("Sr", "In", 2) : 3.519,
    ("Sr", "Sn")    : 3.272,
    ("Sr", "Sn", 2) : 3.347,
    ("Sr", "Sb")    : 3.258,
    ("Sr", "Sb", 2) : 3.157,
    ("Sr", "Te")    : 3.186,
    ("Sr", "Te", 2) : 3.013,
    ("Sr", "I" )    : 3.091,
    ("Sr", "Cs")    : 2.750,
    ("Sr", "Ba")    : 2.750,
    ("Sr", "Ba", 2) : 2.750,
    ("Sr", "La")    : 2.750,
    ("Sr", "Hf")    : 2.750,
    ("Sr", "Hf", 2) : 2.750,
    ("Sr", "Ta")    : 2.750,
    ("Sr", "Ta", 2) : 2.750,
    ("Sr", "W" )    : 2.750,
    ("Sr", "W" , 2) : 2.750,
    ("Sr", "Re")    : 2.750,
    ("Sr", "Re", 2) : 2.750,
    ("Sr", "Os")    : 2.750,
    ("Sr", "Os", 2) : 2.750,
    ("Sr", "Ir")    : 2.750,
    ("Sr", "Ir", 2) : 2.750,
    ("Sr", "Pt")    : 2.750,
    ("Sr", "Pt", 2) : 2.750,
    ("Sr", "Au")    : 2.750,
    ("Sr", "Tl")    : 2.750,
    ("Sr", "Tl", 2) : 2.750,
    ("Sr", "Pb")    : 2.750,
    ("Sr", "Pb", 2) : 2.750,
    ("Sr", "Bi")    : 2.750,
    ("Sr", "Bi", 2) : 2.750,
    ("Sr", "Po")    : 2.750,
    ("Sr", "Po", 2) : 2.750,
    ("Sr", "At")    : 2.750,
    ("Y",  "Y" )    : 2.750,
    ("Y",  "Zr")    : 2.750,
    ("Y",  "Nb")    : 2.750,
    ("Y",  "Mo")    : 2.750,
    ("Y",  "Tc")    : 2.750,
    ("Y",  "Ru")    : 2.750,
    ("Y",  "Rh")    : 2.750,
    ("Y",  "Pd")    : 2.750,
    ("Y",  "Ag")    : 2.750,
    ("Y",  "In")    : 2.750,
    ("Y",  "Sn")    : 2.750,
    ("Y",  "Sb")    : 2.750,
    ("Y",  "Te")    : 2.750,
    ("Y",  "I" )    : 2.750,
    ("Y",  "Cs")    : 2.750,
    ("Y",  "Ba")    : 2.750,
    ("Y",  "La")    : 2.750,
    ("Y",  "Hf")    : 2.750,
    ("Y",  "Ta")    : 2.750,
    ("Y",  "W" )    : 2.750,
    ("Y",  "Re")    : 2.750,
    ("Y",  "Os")    : 2.750,
    ("Y",  "Ir")    : 2.750,
    ("Y",  "Pt")    : 2.750,
    ("Y",  "Au")    : 2.750,
    ("Y",  "Tl")    : 2.750,
    ("Y",  "Pb")    : 2.750,
    ("Y",  "Bi")    : 2.750,
    ("Y",  "Po")    : 2.750,
    ("Y",  "At")    : 2.750,
    ("Zr", "Zr")    : 2.750,
    ("Zr", "Zr", 2) : 2.750,
    ("Zr", "Nb")    : 2.750,
    ("Zr", "Nb", 2) : 2.750,
    ("Zr", "Mo")    : 2.750,
    ("Zr", "Mo", 2) : 2.750,
    ("Zr", "Tc")    : 2.750,
    ("Zr", "Tc", 2) : 2.750,
    ("Zr", "Ru")    : 2.750,
    ("Zr", "Ru", 2) : 2.750,
    ("Zr", "Rh")    : 2.750,
    ("Zr", "Rh", 2) : 2.750,
    ("Zr", "Pd")    : 2.750,
    ("Zr", "Pd", 2) : 2.750,
    ("Zr", "Ag")    : 2.750,
    ("Zr", "In")    : 2.750,
    ("Zr", "In", 2) : 2.750,
    ("Zr", "Sn")    : 2.750,
    ("Zr", "Sn", 2) : 2.750,
    ("Zr", "Sb")    : 2.750,
    ("Zr", "Sb", 2) : 2.750,
    ("Zr", "Te")    : 2.750,
    ("Zr", "Te", 2) : 2.750,
    ("Zr", "I" )    : 2.750,
    ("Zr", "Cs")    : 2.750,
    ("Zr", "Ba")    : 2.750,
    ("Zr", "Ba", 2) : 2.750,
    ("Zr", "La")    : 2.750,
    ("Zr", "Hf")    : 2.750,
    ("Zr", "Hf", 2) : 2.750,
    ("Zr", "Ta")    : 2.750,
    ("Zr", "Ta", 2) : 2.750,
    ("Zr", "W" )    : 2.750,
    ("Zr", "W" , 2) : 2.750,
    ("Zr", "Re")    : 2.750,
    ("Zr", "Re", 2) : 2.750,
    ("Zr", "Os")    : 2.750,
    ("Zr", "Os", 2) : 2.750,
    ("Zr", "Ir")    : 2.750,
    ("Zr", "Ir", 2) : 2.750,
    ("Zr", "Pt")    : 2.750,
    ("Zr", "Pt", 2) : 2.750,
    ("Zr", "Au")    : 2.750,
    ("Zr", "Tl")    : 2.750,
    ("Zr", "Tl", 2) : 2.750,
    ("Zr", "Pb")    : 2.750,
    ("Zr", "Pb", 2) : 2.750,
    ("Zr", "Bi")    : 2.750,
    ("Zr", "Bi", 2) : 2.750,
    ("Zr", "Po")    : 2.750,
    ("Zr", "Po", 2) : 2.750,
    ("Zr", "At")    : 2.750,
    ("Nb", "Nb")    : 2.750,
    ("Nb", "Nb", 2) : 2.750,
    ("Nb", "Mo")    : 2.750,
    ("Nb", "Mo", 2) : 2.750,
    ("Nb", "Tc")    : 2.750,
    ("Nb", "Tc", 2) : 2.750,
    ("Nb", "Ru")    : 2.750,
    ("Nb", "Ru", 2) : 2.750,
    ("Nb", "Rh")    : 2.750,
    ("Nb", "Rh", 2) : 2.750,
    ("Nb", "Pd")    : 2.750,
    ("Nb", "Pd", 2) : 2.750,
    ("Nb", "Ag")    : 2.750,
    ("Nb", "In")    : 2.750,
    ("Nb", "In", 2) : 2.750,
    ("Nb", "Sn")    : 2.750,
    ("Nb", "Sn", 2) : 2.750,
    ("Nb", "Sb")    : 2.750,
    ("Nb", "Sb", 2) : 2.750,
    ("Nb", "Te")    : 2.750,
    ("Nb", "Te", 2) : 2.750,
    ("Nb", "I" )    : 2.750,
    ("Nb", "Cs")    : 2.750,
    ("Nb", "Ba")    : 2.750,
    ("Nb", "Ba", 2) : 2.750,
    ("Nb", "La")    : 2.750,
    ("Nb", "Hf")    : 2.750,
    ("Nb", "Hf", 2) : 2.750,
    ("Nb", "Ta")    : 2.750,
    ("Nb", "Ta", 2) : 2.750,
    ("Nb", "W" )    : 2.750,
    ("Nb", "W" , 2) : 2.750,
    ("Nb", "Re")    : 2.750,
    ("Nb", "Re", 2) : 2.750,
    ("Nb", "Os")    : 2.750,
    ("Nb", "Os", 2) : 2.750,
    ("Nb", "Ir")    : 2.750,
    ("Nb", "Ir", 2) : 2.750,
    ("Nb", "Pt")    : 2.750,
    ("Nb", "Pt", 2) : 2.750,
    ("Nb", "Au")    : 2.750,
    ("Nb", "Tl")    : 2.750,
    ("Nb", "Tl", 2) : 2.750,
    ("Nb", "Pb")    : 2.750,
    ("Nb", "Pb", 2) : 2.750,
    ("Nb", "Bi")    : 2.750,
    ("Nb", "Bi", 2) : 2.750,
    ("Nb", "Po")    : 2.750,
    ("Nb", "Po", 2) : 2.750,
    ("Nb", "At")    : 2.750,
    ("Mo", "Mo")    : 2.750,
    ("Mo", "Mo", 2) : 2.750,
    ("Mo", "Tc")    : 2.750,
    ("Mo", "Tc", 2) : 2.750,
    ("Mo", "Ru")    : 2.750,
    ("Mo", "Ru", 2) : 2.750,
    ("Mo", "Rh")    : 2.750,
    ("Mo", "Rh", 2) : 2.750,
    ("Mo", "Pd")    : 2.750,
    ("Mo", "Pd", 2) : 2.750,
    ("Mo", "Ag")    : 2.750,
    ("Mo", "In")    : 2.750,
    ("Mo", "In", 2) : 2.750,
    ("Mo", "Sn")    : 2.750,
    ("Mo", "Sn", 2) : 2.750,
    ("Mo", "Sb")    : 2.750,
    ("Mo", "Sb", 2) : 2.750,
    ("Mo", "Te")    : 2.750,
    ("Mo", "Te", 2) : 2.750,
    ("Mo", "I" )    : 2.750,
    ("Mo", "Cs")    : 2.750,
    ("Mo", "Ba")    : 2.750,
    ("Mo", "Ba", 2) : 2.750,
    ("Mo", "La")    : 2.750,
    ("Mo", "Hf")    : 2.750,
    ("Mo", "Hf", 2) : 2.750,
    ("Mo", "Ta")    : 2.750,
    ("Mo", "Ta", 2) : 2.750,
    ("Mo", "W" )    : 2.750,
    ("Mo", "W" , 2) : 2.750,
    ("Mo", "Re")    : 2.750,
    ("Mo", "Re", 2) : 2.750,
    ("Mo", "Os")    : 2.750,
    ("Mo", "Os", 2) : 2.750,
    ("Mo", "Ir")    : 2.750,
    ("Mo", "Ir", 2) : 2.750,
    ("Mo", "Pt")    : 2.750,
    ("Mo", "Pt", 2) : 2.750,
    ("Mo", "Au")    : 2.750,
    ("Mo", "Tl")    : 2.750,
    ("Mo", "Tl", 2) : 2.750,
    ("Mo", "Pb")    : 2.750,
    ("Mo", "Pb", 2) : 2.750,
    ("Mo", "Bi")    : 2.750,
    ("Mo", "Bi", 2) : 2.750,
    ("Mo", "Po")    : 2.750,
    ("Mo", "Po", 2) : 2.750,
    ("Mo", "At")    : 2.750,
    ("Tc", "Tc")    : 2.750,
    ("Tc", "Tc", 2) : 2.750,
    ("Tc", "Ru")    : 2.750,
    ("Tc", "Ru", 2) : 2.750,
    ("Tc", "Rh")    : 2.750,
    ("Tc", "Rh", 2) : 2.750,
    ("Tc", "Pd")    : 2.750,
    ("Tc", "Pd", 2) : 2.750,
    ("Tc", "Ag")    : 2.750,
    ("Tc", "In")    : 2.750,
    ("Tc", "In", 2) : 2.750,
    ("Tc", "Sn")    : 2.750,
    ("Tc", "Sn", 2) : 2.750,
    ("Tc", "Sb")    : 2.750,
    ("Tc", "Sb", 2) : 2.750,
    ("Tc", "Te")    : 2.750,
    ("Tc", "Te", 2) : 2.750,
    ("Tc", "I" )    : 2.750,
    ("Tc", "Cs")    : 2.750,
    ("Tc", "Ba")    : 2.750,
    ("Tc", "Ba", 2) : 2.750,
    ("Tc", "La")    : 2.750,
    ("Tc", "Hf")    : 2.750,
    ("Tc", "Hf", 2) : 2.750,
    ("Tc", "Ta")    : 2.750,
    ("Tc", "Ta", 2) : 2.750,
    ("Tc", "W" )    : 2.750,
    ("Tc", "W" , 2) : 2.750,
    ("Tc", "Re")    : 2.750,
    ("Tc", "Re", 2) : 2.750,
    ("Tc", "Os")    : 2.750,
    ("Tc", "Os", 2) : 2.750,
    ("Tc", "Ir")    : 2.750,
    ("Tc", "Ir", 2) : 2.750,
    ("Tc", "Pt")    : 2.750,
    ("Tc", "Pt", 2) : 2.750,
    ("Tc", "Au")    : 2.750,
    ("Tc", "Tl")    : 2.750,
    ("Tc", "Tl", 2) : 2.750,
    ("Tc", "Pb")    : 2.750,
    ("Tc", "Pb", 2) : 2.750,
    ("Tc", "Bi")    : 2.750,
    ("Tc", "Bi", 2) : 2.750,
    ("Tc", "Po")    : 2.750,
    ("Tc", "Po", 2) : 2.750,
    ("Tc", "At")    : 2.750,
    ("Ru", "Ru")    : 2.750,
    ("Ru", "Ru", 2) : 2.750,
    ("Ru", "Rh")    : 2.750,
    ("Ru", "Rh", 2) : 2.750,
    ("Ru", "Pd")    : 2.750,
    ("Ru", "Pd", 2) : 2.750,
    ("Ru", "Ag")    : 2.750,
    ("Ru", "In")    : 2.750,
    ("Ru", "In", 2) : 2.750,
    ("Ru", "Sn")    : 2.750,
    ("Ru", "Sn", 2) : 2.750,
    ("Ru", "Sb")    : 2.750,
    ("Ru", "Sb", 2) : 2.750,
    ("Ru", "Te")    : 2.750,
    ("Ru", "Te", 2) : 2.750,
    ("Ru", "I" )    : 2.750,
    ("Ru", "Cs")    : 2.750,
    ("Ru", "Ba")    : 2.750,
    ("Ru", "Ba", 2) : 2.750,
    ("Ru", "La")    : 2.750,
    ("Ru", "Hf")    : 2.750,
    ("Ru", "Hf", 2) : 2.750,
    ("Ru", "Ta")    : 2.750,
    ("Ru", "Ta", 2) : 2.750,
    ("Ru", "W" )    : 2.750,
    ("Ru", "W" , 2) : 2.750,
    ("Ru", "Re")    : 2.750,
    ("Ru", "Re", 2) : 2.750,
    ("Ru", "Os")    : 2.750,
    ("Ru", "Os", 2) : 2.750,
    ("Ru", "Ir")    : 2.750,
    ("Ru", "Ir", 2) : 2.750,
    ("Ru", "Pt")    : 2.750,
    ("Ru", "Pt", 2) : 2.750,
    ("Ru", "Au")    : 2.750,
    ("Ru", "Tl")    : 2.750,
    ("Ru", "Tl", 2) : 2.750,
    ("Ru", "Pb")    : 2.750,
    ("Ru", "Pb", 2) : 2.750,
    ("Ru", "Bi")    : 2.750,
    ("Ru", "Bi", 2) : 2.750,
    ("Ru", "Po")    : 2.750,
    ("Ru", "Po", 2) : 2.750,
    ("Ru", "At")    : 2.750,
    ("Rh", "Rh")    : 2.750,
    ("Rh", "Rh", 2) : 2.750,
    ("Rh", "Pd")    : 2.750,
    ("Rh", "Pd", 2) : 2.750,
    ("Rh", "Ag")    : 2.750,
    ("Rh", "In")    : 2.750,
    ("Rh", "In", 2) : 2.750,
    ("Rh", "Sn")    : 2.750,
    ("Rh", "Sn", 2) : 2.750,
    ("Rh", "Sb")    : 2.750,
    ("Rh", "Sb", 2) : 2.750,
    ("Rh", "Te")    : 2.750,
    ("Rh", "Te", 2) : 2.750,
    ("Rh", "I" )    : 2.750,
    ("Rh", "Cs")    : 2.750,
    ("Rh", "Ba")    : 2.750,
    ("Rh", "Ba", 2) : 2.750,
    ("Rh", "La")    : 2.750,
    ("Rh", "Hf")    : 2.750,
    ("Rh", "Hf", 2) : 2.750,
    ("Rh", "Ta")    : 2.750,
    ("Rh", "Ta", 2) : 2.750,
    ("Rh", "W" )    : 2.750,
    ("Rh", "W" , 2) : 2.750,
    ("Rh", "Re")    : 2.750,
    ("Rh", "Re", 2) : 2.750,
    ("Rh", "Os")    : 2.750,
    ("Rh", "Os", 2) : 2.750,
    ("Rh", "Ir")    : 2.750,
    ("Rh", "Ir", 2) : 2.750,
    ("Rh", "Pt")    : 2.750,
    ("Rh", "Pt", 2) : 2.750,
    ("Rh", "Au")    : 2.750,
    ("Rh", "Tl")    : 2.750,
    ("Rh", "Tl", 2) : 2.750,
    ("Rh", "Pb")    : 2.750,
    ("Rh", "Pb", 2) : 2.750,
    ("Rh", "Bi")    : 2.750,
    ("Rh", "Bi", 2) : 2.750,
    ("Rh", "Po")    : 2.750,
    ("Rh", "Po", 2) : 2.750,
    ("Rh", "At")    : 2.750,
    ("Pd", "Pd")    : 2.750,
    ("Pd", "Pd", 2) : 2.750,
    ("Pd", "Ag")    : 2.750,
    ("Pd", "In")    : 2.750,
    ("Pd", "In", 2) : 2.750,
    ("Pd", "Sn")    : 2.750,
    ("Pd", "Sn", 2) : 2.750,
    ("Pd", "Sb")    : 2.750,
    ("Pd", "Sb", 2) : 2.750,
    ("Pd", "Te")    : 2.750,
    ("Pd", "Te", 2) : 2.750,
    ("Pd", "I" )    : 2.750,
    ("Pd", "Cs")    : 2.750,
    ("Pd", "Ba")    : 2.750,
    ("Pd", "Ba", 2) : 2.750,
    ("Pd", "La")    : 2.750,
    ("Pd", "Hf")    : 2.750,
    ("Pd", "Hf", 2) : 2.750,
    ("Pd", "Ta")    : 2.750,
    ("Pd", "Ta", 2) : 2.750,
    ("Pd", "W" )    : 2.750,
    ("Pd", "W" , 2) : 2.750,
    ("Pd", "Re")    : 2.750,
    ("Pd", "Re", 2) : 2.750,
    ("Pd", "Os")    : 2.750,
    ("Pd", "Os", 2) : 2.750,
    ("Pd", "Ir")    : 2.750,
    ("Pd", "Ir", 2) : 2.750,
    ("Pd", "Pt")    : 2.750,
    ("Pd", "Pt", 2) : 2.750,
    ("Pd", "Au")    : 2.750,
    ("Pd", "Tl")    : 2.750,
    ("Pd", "Tl", 2) : 2.750,
    ("Pd", "Pb")    : 2.750,
    ("Pd", "Pb", 2) : 2.750,
    ("Pd", "Bi")    : 2.750,
    ("Pd", "Bi", 2) : 2.750,
    ("Pd", "Po")    : 2.750,
    ("Pd", "Po", 2) : 2.750,
    ("Pd", "At")    : 2.750,
    ("Ag", "Ag")    : 2.750,
    ("Ag", "In")    : 2.750,
    ("Ag", "Sn")    : 2.750,
    ("Ag", "Sb")    : 2.750,
    ("Ag", "Te")    : 2.750,
    ("Ag", "I" )    : 2.750,
    ("Ag", "Cs")    : 2.750,
    ("Ag", "Ba")    : 2.750,
    ("Ag", "La")    : 2.750,
    ("Ag", "Hf")    : 2.750,
    ("Ag", "Ta")    : 2.750,
    ("Ag", "W" )    : 2.750,
    ("Ag", "Re")    : 2.750,
    ("Ag", "Os")    : 2.750,
    ("Ag", "Ir")    : 2.750,
    ("Ag", "Pt")    : 2.750,
    ("Ag", "Au")    : 2.750,
    ("Ag", "Tl")    : 2.750,
    ("Ag", "Pb")    : 2.750,
    ("Ag", "Bi")    : 2.750,
    ("Ag", "Po")    : 2.750,
    ("Ag", "At")    : 2.750,
    ("In", "In")    : 2.678,
    ("In", "In", 2) : 2.358,
    ("In", "Sn")    : 2.770,
    ("In", "Sn", 2) : 2.438,
    ("In", "Sb")    : 2.793,
    ("In", "Sb", 2) : 2.448,
    ("In", "Te")    : 2.681,
    ("In", "Te", 2) : 2.383,
    ("In", "I" )    : 2.651,
    ("In", "Cs")    : 2.750,
    ("In", "Ba")    : 2.750,
    ("In", "Ba", 2) : 2.750,
    ("In", "La")    : 2.750,
    ("In", "Hf")    : 2.750,
    ("In", "Hf", 2) : 2.750,
    ("In", "Ta")    : 2.750,
    ("In", "Ta", 2) : 2.750,
    ("In", "W" )    : 2.750,
    ("In", "W" , 2) : 2.750,
    ("In", "Re")    : 2.750,
    ("In", "Re", 2) : 2.750,
    ("In", "Os")    : 2.750,
    ("In", "Os", 2) : 2.750,
    ("In", "Ir")    : 2.750,
    ("In", "Ir", 2) : 2.750,
    ("In", "Pt")    : 2.750,
    ("In", "Pt", 2) : 2.750,
    ("In", "Au")    : 2.750,
    ("In", "Tl")    : 2.750,
    ("In", "Tl", 2) : 2.750,
    ("In", "Pb")    : 2.750,
    ("In", "Pb", 2) : 2.750,
    ("In", "Bi")    : 2.750,
    ("In", "Bi", 2) : 2.750,
    ("In", "Po")    : 2.750,
    ("In", "Po", 2) : 2.750,
    ("In", "At")    : 2.750,
    ("Sn", "Sn")    : 2.778,
    ("Sn", "Sn", 2) : 2.469,
    ("Sn", "Sb")    : 2.786,
    ("Sn", "Sb", 2) : 2.516,
    ("Sn", "Te")    : 2.743,
    ("Sn", "Te", 2) : 2.486,
    ("Sn", "I" )    : 2.690,
    ("Sn", "Cs")    : 2.750,
    ("Sn", "Ba")    : 2.750,
    ("Sn", "Ba", 2) : 2.750,
    ("Sn", "La")    : 2.750,
    ("Sn", "Hf")    : 2.750,
    ("Sn", "Hf", 2) : 2.750,
    ("Sn", "Ta")    : 2.750,
    ("Sn", "Ta", 2) : 2.750,
    ("Sn", "W" )    : 2.750,
    ("Sn", "W" , 2) : 2.750,
    ("Sn", "Re")    : 2.750,
    ("Sn", "Re", 2) : 2.750,
    ("Sn", "Os")    : 2.750,
    ("Sn", "Os", 2) : 2.750,
    ("Sn", "Ir")    : 2.750,
    ("Sn", "Ir", 2) : 2.750,
    ("Sn", "Pt")    : 2.750,
    ("Sn", "Pt", 2) : 2.750,
    ("Sn", "Au")    : 2.750,
    ("Sn", "Tl")    : 2.750,
    ("Sn", "Tl", 2) : 2.750,
    ("Sn", "Pb")    : 2.750,
    ("Sn", "Pb", 2) : 2.750,
    ("Sn", "Bi")    : 2.750,
    ("Sn", "Bi", 2) : 2.750,
    ("Sn", "Po")    : 2.750,
    ("Sn", "Po", 2) : 2.750,
    ("Sn", "At")    : 2.750,
    ("Sb", "Sb")    : 2.803,
    ("Sb", "Sb", 2) : 2.566,
    ("Sb", "Te")    : 2.762,
    ("Sb", "Te", 2) : 2.548,
    ("Sb", "I" )    : 2.729,
    ("Sb", "Cs")    : 2.750,
    ("Sb", "Ba")    : 2.750,
    ("Sb", "Ba", 2) : 2.750,
    ("Sb", "La")    : 2.750,
    ("Sb", "Hf")    : 2.750,
    ("Sb", "Hf", 2) : 2.750,
    ("Sb", "Ta")    : 2.750,
    ("Sb", "Ta", 2) : 2.750,
    ("Sb", "W" )    : 2.750,
    ("Sb", "W" , 2) : 2.750,
    ("Sb", "Re")    : 2.750,
    ("Sb", "Re", 2) : 2.750,
    ("Sb", "Os")    : 2.750,
    ("Sb", "Os", 2) : 2.750,
    ("Sb", "Ir")    : 2.750,
    ("Sb", "Ir", 2) : 2.750,
    ("Sb", "Pt")    : 2.750,
    ("Sb", "Pt", 2) : 2.750,
    ("Sb", "Au")    : 2.750,
    ("Sb", "Tl")    : 2.750,
    ("Sb", "Tl", 2) : 2.750,
    ("Sb", "Pb")    : 2.750,
    ("Sb", "Pb", 2) : 2.750,
    ("Sb", "Bi")    : 2.750,
    ("Sb", "Bi", 2) : 2.750,
    ("Sb", "Po")    : 2.750,
    ("Sb", "Po", 2) : 2.750,
    ("Sb", "At")    : 2.750,
    ("Te", "Te")    : 2.774,
    ("Te", "Te", 2) : 2.569,
    ("Te", "I" )    : 2.728,
    ("Te", "Cs")    : 2.750,
    ("Te", "Ba")    : 2.750,
    ("Te", "Ba", 2) : 2.750,
    ("Te", "La")    : 2.750,
    ("Te", "Hf")    : 2.750,
    ("Te", "Hf", 2) : 2.750,
    ("Te", "Ta")    : 2.750,
    ("Te", "Ta", 2) : 2.750,
    ("Te", "W" )    : 2.750,
    ("Te", "W" , 2) : 2.750,
    ("Te", "Re")    : 2.750,
    ("Te", "Re", 2) : 2.750,
    ("Te", "Os")    : 2.750,
    ("Te", "Os", 2) : 2.750,
    ("Te", "Ir")    : 2.750,
    ("Te", "Ir", 2) : 2.750,
    ("Te", "Pt")    : 2.750,
    ("Te", "Pt", 2) : 2.750,
    ("Te", "Au")    : 2.750,
    ("Te", "Tl")    : 2.750,
    ("Te", "Tl", 2) : 2.750,
    ("Te", "Pb")    : 2.750,
    ("Te", "Pb", 2) : 2.750,
    ("Te", "Bi")    : 2.750,
    ("Te", "Bi", 2) : 2.750,
    ("Te", "Po")    : 2.750,
    ("Te", "Po", 2) : 2.750,
    ("Te", "At")    : 2.750,
    ("I",  "I" )    : 2.704,
    ("I",  "Cs")    : 2.750,
    ("I",  "Ba")    : 2.750,
    ("I",  "La")    : 2.750,
    ("I",  "Hf")    : 2.750,
    ("I",  "Ta")    : 2.750,
    ("I",  "W" )    : 2.750,
    ("I",  "Re")    : 2.750,
    ("I",  "Os")    : 2.750,
    ("I",  "Ir")    : 2.750,
    ("I",  "Pt")    : 2.750,
    ("I",  "Au")    : 2.750,
    ("I",  "Tl")    : 2.750,
    ("I",  "Pb")    : 2.750,
    ("I",  "Bi")    : 2.750,
    ("I",  "Po")    : 2.750,
    ("I",  "At")    : 2.750,
    ("Cs", "Cs")    : 2.750,
    ("Cs", "Ba")    : 2.750,
    ("Cs", "La")    : 2.750,
    ("Cs", "Hf")    : 2.750,
    ("Cs", "Ta")    : 2.750,
    ("Cs", "W" )    : 2.750,
    ("Cs", "Re")    : 2.750,
    ("Cs", "Os")    : 2.750,
    ("Cs", "Ir")    : 2.750,
    ("Cs", "Pt")    : 2.750,
    ("Cs", "Au")    : 2.750,
    ("Cs", "Tl")    : 2.750,
    ("Cs", "Pb")    : 2.750,
    ("Cs", "Bi")    : 2.750,
    ("Cs", "Po")    : 2.750,
    ("Cs", "At")    : 2.750,
    ("Ba", "Ba")    : 2.750,
    ("Ba", "Ba", 2) : 2.750,
    ("Ba", "La")    : 2.750,
    ("Ba", "Hf")    : 2.750,
    ("Ba", "Hf", 2) : 2.750,
    ("Ba", "Ta")    : 2.750,
    ("Ba", "Ta", 2) : 2.750,
    ("Ba", "W" )    : 2.750,
    ("Ba", "W" , 2) : 2.750,
    ("Ba", "Re")    : 2.750,
    ("Ba", "Re", 2) : 2.750,
    ("Ba", "Os")    : 2.750,
    ("Ba", "Os", 2) : 2.750,
    ("Ba", "Ir")    : 2.750,
    ("Ba", "Ir", 2) : 2.750,
    ("Ba", "Pt")    : 2.750,
    ("Ba", "Pt", 2) : 2.750,
    ("Ba", "Au")    : 2.750,
    ("Ba", "Tl")    : 2.750,
    ("Ba", "Tl", 2) : 2.750,
    ("Ba", "Pb")    : 2.750,
    ("Ba", "Pb", 2) : 2.750,
    ("Ba", "Bi")    : 2.750,
    ("Ba", "Bi", 2) : 2.750,
    ("Ba", "Po")    : 2.750,
    ("Ba", "Po", 2) : 2.750,
    ("Ba", "At")    : 2.750,
    ("La", "La")    : 2.750,
    ("La", "Hf")    : 2.750,
    ("La", "Ta")    : 2.750,
    ("La", "W" )    : 2.750,
    ("La", "Re")    : 2.750,
    ("La", "Os")    : 2.750,
    ("La", "Ir")    : 2.750,
    ("La", "Pt")    : 2.750,
    ("La", "Au")    : 2.750,
    ("La", "Tl")    : 2.750,
    ("La", "Pb")    : 2.750,
    ("La", "Bi")    : 2.750,
    ("La", "Po")    : 2.750,
    ("La", "At")    : 2.750,
    ("Hf", "Hf")    : 2.750,
    ("Hf", "Hf", 2) : 2.750,
    ("Hf", "Ta")    : 2.750,
    ("Hf", "Ta", 2) : 2.750,
    ("Hf", "W" )    : 2.750,
    ("Hf", "W" , 2) : 2.750,
    ("Hf", "Re")    : 2.750,
    ("Hf", "Re", 2) : 2.750,
    ("Hf", "Os")    : 2.750,
    ("Hf", "Os", 2) : 2.750,
    ("Hf", "Ir")    : 2.750,
    ("Hf", "Ir", 2) : 2.750,
    ("Hf", "Pt")    : 2.750,
    ("Hf", "Pt", 2) : 2.750,
    ("Hf", "Au")    : 2.750,
    ("Hf", "Tl")    : 2.750,
    ("Hf", "Tl", 2) : 2.750,
    ("Hf", "Pb")    : 2.750,
    ("Hf", "Pb", 2) : 2.750,
    ("Hf", "Bi")    : 2.750,
    ("Hf", "Bi", 2) : 2.750,
    ("Hf", "Po")    : 2.750,
    ("Hf", "Po", 2) : 2.750,
    ("Hf", "At")    : 2.750,
    ("Ta", "Ta")    : 2.750,
    ("Ta", "Ta", 2) : 2.750,
    ("Ta", "W" )    : 2.750,
    ("Ta", "W" , 2) : 2.750,
    ("Ta", "Re")    : 2.750,
    ("Ta", "Re", 2) : 2.750,
    ("Ta", "Os")    : 2.750,
    ("Ta", "Os", 2) : 2.750,
    ("Ta", "Ir")    : 2.750,
    ("Ta", "Ir", 2) : 2.750,
    ("Ta", "Pt")    : 2.750,
    ("Ta", "Pt", 2) : 2.750,
    ("Ta", "Au")    : 2.750,
    ("Ta", "Tl")    : 2.750,
    ("Ta", "Tl", 2) : 2.750,
    ("Ta", "Pb")    : 2.750,
    ("Ta", "Pb", 2) : 2.750,
    ("Ta", "Bi")    : 2.750,
    ("Ta", "Bi", 2) : 2.750,
    ("Ta", "Po")    : 2.750,
    ("Ta", "Po", 2) : 2.750,
    ("Ta", "At")    : 2.750,
    ("W",  "W" )    : 2.750,
    ("W",  "W" , 2) : 2.750,
    ("W",  "Re")    : 2.750,
    ("W",  "Re", 2) : 2.750,
    ("W",  "Os")    : 2.750,
    ("W",  "Os", 2) : 2.750,
    ("W",  "Ir")    : 2.750,
    ("W",  "Ir", 2) : 2.750,
    ("W",  "Pt")    : 2.750,
    ("W",  "Pt", 2) : 2.750,
    ("W",  "Au")    : 2.750,
    ("W",  "Tl")    : 2.750,
    ("W",  "Tl", 2) : 2.750,
    ("W",  "Pb")    : 2.750,
    ("W",  "Pb", 2) : 2.750,
    ("W",  "Bi")    : 2.750,
    ("W",  "Bi", 2) : 2.750,
    ("W",  "Po")    : 2.750,
    ("W",  "Po", 2) : 2.750,
    ("W",  "At")    : 2.750,
    ("Re", "Re")    : 2.750,
    ("Re", "Re", 2) : 2.750,
    ("Re", "Os")    : 2.750,
    ("Re", "Os", 2) : 2.750,
    ("Re", "Ir")    : 2.750,
    ("Re", "Ir", 2) : 2.750,
    ("Re", "Pt")    : 2.750,
    ("Re", "Pt", 2) : 2.750,
    ("Re", "Au")    : 2.750,
    ("Re", "Tl")    : 2.750,
    ("Re", "Tl", 2) : 2.750,
    ("Re", "Pb")    : 2.750,
    ("Re", "Pb", 2) : 2.750,
    ("Re", "Bi")    : 2.750,
    ("Re", "Bi", 2) : 2.750,
    ("Re", "Po")    : 2.750,
    ("Re", "Po", 2) : 2.750,
    ("Re", "At")    : 2.750,
    ("Os", "Os")    : 2.750,
    ("Os", "Os", 2) : 2.750,
    ("Os", "Ir")    : 2.750,
    ("Os", "Ir", 2) : 2.750,
    ("Os", "Pt")    : 2.750,
    ("Os", "Pt", 2) : 2.750,
    ("Os", "Au")    : 2.750,
    ("Os", "Tl")    : 2.750,
    ("Os", "Tl", 2) : 2.750,
    ("Os", "Pb")    : 2.750,
    ("Os", "Pb", 2) : 2.750,
    ("Os", "Bi")    : 2.750,
    ("Os", "Bi", 2) : 2.750,
    ("Os", "Po")    : 2.750,
    ("Os", "Po", 2) : 2.750,
    ("Os", "At")    : 2.750,
    ("Ir", "Ir")    : 2.750,
    ("Ir", "Ir", 2) : 2.750,
    ("Ir", "Pt")    : 2.750,
    ("Ir", "Pt", 2) : 2.750,
    ("Ir", "Au")    : 2.750,
    ("Ir", "Tl")    : 2.750,
    ("Ir", "Tl", 2) : 2.750,
    ("Ir", "Pb")    : 2.750,
    ("Ir", "Pb", 2) : 2.750,
    ("Ir", "Bi")    : 2.750,
    ("Ir", "Bi", 2) : 2.750,
    ("Ir", "Po")    : 2.750,
    ("Ir", "Po", 2) : 2.750,
    ("Ir", "At")    : 2.750,
    ("Pt", "Pt")    : 2.750,
    ("Pt", "Pt", 2) : 2.750,
    ("Pt", "Au")    : 2.750,
    ("Pt", "Tl")    : 2.750,
    ("Pt", "Tl", 2) : 2.750,
    ("Pt", "Pb")    : 2.750,
    ("Pt", "Pb", 2) : 2.750,
    ("Pt", "Bi")    : 2.750,
    ("Pt", "Bi", 2) : 2.750,
    ("Pt", "Po")    : 2.750,
    ("Pt", "Po", 2) : 2.750,
    ("Pt", "At")    : 2.750,
    ("Au", "Au")    : 2.750,
    ("Au", "Tl")    : 2.750,
    ("Au", "Pb")    : 2.750,
    ("Au", "Bi")    : 2.750,
    ("Au", "Po")    : 2.750,
    ("Au", "At")    : 2.750,
    ("Tl", "Tl")    : 2.750,
    ("Tl", "Tl", 2) : 2.750,
    ("Tl", "Pb")    : 2.750,
    ("Tl", "Pb", 2) : 2.750,
    ("Tl", "Bi")    : 2.750,
    ("Tl", "Bi", 2) : 2.750,
    ("Tl", "Po")    : 2.750,
    ("Tl", "Po", 2) : 2.750,
    ("Tl", "At")    : 2.750,
    ("Pb", "Pb")    : 2.750,
    ("Pb", "Pb", 2) : 2.750,
    ("Pb", "Bi")    : 2.750,
    ("Pb", "Bi", 2) : 2.750,
    ("Pb", "Po")    : 2.750,
    ("Pb", "Po", 2) : 2.750,
    ("Pb", "At")    : 2.750,
    ("Bi", "Bi")    : 2.750,
    ("Bi", "Bi", 2) : 2.750,
    ("Bi", "Po")    : 2.750,
    ("Bi", "Po", 2) : 2.750,
    ("Bi", "At")    : 2.750,
    ("Po", "Po")    : 2.750,
    ("Po", "Po", 2) : 2.750,
    ("Po", "At")    : 2.750,
    ("At", "At")    : 2.750,
  }

def get_default_bondlength(s1, s2, order=1):
  if len(s1)==1: s1=s1.upper()
  if len(s2)==1: s2=s2.upper()
  s1=s1.strip()
  s2=s2.strip()
  order = int(order)
  for key in [
    (s1, s2, order),
    (s2, s1, order),
    (s1, s2),
    (s2, s1),
    ]:
    if key in qm_defaults:
      return qm_defaults[key]
  if None:
    if s1 in ["H", "D", "T"] or s2 in ["H", "D", "T"]:
      return 1.5
    else:
      return 2.75
  return None

def generic_metal_run(metal_element,
                      coordination_atom_element,
                      resName,
                      name,
                      return_non_protein=True,
                      ):
  coordination_atom_element = coordination_atom_element.strip()
  local_metal_defaults = metal_defaults.get(metal_element.capitalize(), None)
  if local_metal_defaults is None: return None
  if coordination_atom_element=="O":
    # is this too harsh? excludes non-standard aminoacids
    if resName.upper() in three_letter_given_one_letter.values():
      if name and name=="O":
        result = carbonyl.get(metal_element.capitalize(), None)
        return result
  element_defaults = local_metal_defaults.get(coordination_atom_element,
                                        None,
                                        )
  if element_defaults is None: return None
  residue_defaults = element_defaults.get(resName.upper(),
                                        None,
                                        )
  if residue_defaults is None:
    if return_non_protein:
      result = non_protein.get((metal_element.capitalize(),
                                coordination_atom_element,
                                ),
                               None,
                               )
      return result
    else:
      return None
  return residue_defaults

basepairs_lengths = {

# taken from base_pairing.data

# columns:
# pair_type   saenger   paired_bases    hydrogen_flag   atom_pair   [atom_pair ...]
# atom_pair = atom1, atom2, bond_length, standard_deviation, slack
# key:value =
# key=saenger class
# value = [(bp1, bp2), (atom1, atom2, target distance, sigma, slack), (another link),]
#
# Canonical Watson-Crick base pairings
#homo purine
1:[('A','A'), ("N6","N1",2.99,0.14,0.07) ],
2:[('A','A'), ("N6","N7",2.95,0.20,0.06) ],
3:[('G','G'), ("N1","O6",2.86,0.12,0.01) ],
4:[('G','G'), ("N2","N3",3.10,0.22,0.13) ],
5:[('A','A'), ("N6","N1",2.93,0.18,0.  ), ("N7","N6",3.07,0.17,0.  ) ],
6:[('G','G'), ("O6","N1",2.88,0.16,0.  ), ("N7","N2",2.91,0.15,0.  ) ],
7:[('G','G'), ("N7","N1",2.93,0.17,0.  ), ("O6","N2",2.76,0.24,0.  ) ],
8:[('A','G'), ("N6","O6",2.95,0.21,0.  ), ("N1","N1",2.88,0.15,0.  ) ],
9:[('A','G'), ("N6","O6",2.70,0.08,0.  ), ("N7","N1",3.08,0.08,0.  ) ],
10:[('A','G'), ("N6","N3",3.15,0.11,0.  ), ("N1","N2",3.02,0.19,0.  ) ],
11:[('A','G'), ("N6","N3",3.23,0.15,0.  ), ("N7","N2",3.00,0.16,0.  ) ],
12:[('U','U'), ("N3","O4",2.86,0.05,0.12) ],
13:[('U','U'), ("O2","N3",2.74,0.05,0.03) ],
14:[('C','C'), ("N3","N4",2.72,0.05,0.6 ) ],
15:[('C','C'), ("O2","N4",2.83,0.01,0.  ) ],
17:[('C','U'), ("N3","N3",2.98,0.05,0.  ), ("N4","O2",2.91,0.07,0.  ) ],
18:[('C','U'), ("N4","O4",3.19,0.07,0.  ), ("N3","N3",3.16,0.06,0.  ) ],
19:[('C','G'), ("N4","O6",2.96,0.17,0.  ), ("N3","N1",2.89,0.11,0.  ), ("O2","N2",2.77,0.15, 0.)],
20:[('A','U'), ("N6","O4",3.00,0.17,0.  ), ("N1","N3",2.84,0.12,0.  ) ],
21:[('A','U'), ("N6","O2",2.94,0.17,0.  ), ("N1","N3",2.84,0.13,0.  ) ],
22:[('C','G'), ("O2","N1",2.80,0.17,0.  ), ("N3","N2",2.86,0.18,0.  ) ],
23:[('A','U'), ("N6","O4",3.05,0.15,0.  ), ("N7","N3",2.96,0.15,0.  ) ],
24:[('A','U'), ("N6","O2",2.91,0.19,0.  ), ("N7","N3",2.87,0.13,0.  ) ],
25:[('A','C'), ("N6","N3",3.13,0.15,0.  ), ("N7","N4",3.08,0.18,0.  ) ],
26:[('A','C'), ("N6","N3",3.09,0.14,0.  ), ("N1","N4",3.01,0.08,0.  ) ],
27:[('G','U'), ("O6","N3",2.99,0.15,0.  ), ("N1","O4",2.99,0.07,0.  ) ],
28:[('G','U'), ("O6","N3",2.83,0.13,0.  ), ("N1","O2",2.79,0.13,0.  ) ],
}

rome_numbers = ["","I", "II", "III","IV","V","VI","VII","VIII","IX","X",
  "XI","XII", "XIII","XIV","XV","XVI","XVII","XVIII","XIX","XX",
  "XXI","XXII","XXIII","XXIV", "XXV","XXVI","XXVII","XXVIII"]

def get_basepair_target_distance(atom1, atom2, which_basepair):
  an1 = atom1.name.strip()
  an2 = atom2.name.strip()
  for link in basepairs_lengths[which_basepair]:
    if (an1, an2) == link[:2] or (an2, an1) == link[:2]:
      return link[2:]
  return [2.90, 0.2, 0.0] # default - weak hbond. It returns when basepair
                        # is so distorted that wrong atoms seems to be linked

def run(atom1, atom2, return_non_protein=True):
  from mmtbx.monomer_library.linking_utils import ad_hoc_single_metal_residue_element_types
  metal = None
  coordination_atom = None
  if atom1.element.upper().strip() in ad_hoc_single_metal_residue_element_types:
    metal = atom1
    coordination_atom = atom2
  elif atom2.element.upper().strip() in ad_hoc_single_metal_residue_element_types:
    metal = atom2
    coordination_atom = atom1
  rc = None
  if metal is not None:
    if not hasattr(metal, "symbol"): # iotbx atom
      rc = generic_metal_run(metal.element.strip(),
                             coordination_atom.element.strip(),
                             coordination_atom.parent().resname,
                             getattr(coordination_atom, "name", None),
                             return_non_protein=return_non_protein,
        )
    else:
      rc = generic_metal_run(metal.symbol,
                             coordination_atom.symbol,
                             coordination_atom.resName,
                             getattr(coordination_atom, "name", None),
                             return_non_protein=return_non_protein,
        )
  if rc is None:
    if not hasattr(atom1, "symbol"): # iotbx atom
      return get_default_bondlength(atom1.element.strip(),
                                    atom2.element.strip(),
                                    ), 0.01
    else:
      return get_default_bondlength(atom1.symbol,
                                    atom2.symbol,
                                    ), 0.01
  else: return rc

if __name__=="__main__":
  results = [
    [2.41, 0.1], # Na
    [2.41, 0.1],
    [2.41, 0.1],
    [2.38, 0.1],
    [2.43, non_protein_sigma],
    [2.43, non_protein_sigma],
    [2.31, 0.1],
    [2.38, 0.1],
    [2.09, 0.05], # Zn
    [2.09, 0.05],
    [1.99, 0.05],
    [2.07, 0.25],
    [2.01, non_protein_sigma],
    [2.01, non_protein_sigma],
    [1.89, 0.05],
    [2.07, 0.25],
    None, # V
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    ]
  i=0
  for m in ["Na", "Zn", "V"]:
    for c in ["O", "N", "B"]:
      for residue in ["HOH", "ASP", "LIG", "TYR"]:
        for name in [None, "O"]:
          if c!="O": continue
          metal = AtomClass(m)
          coordination_atom = AtomClass(c)
          coordination_atom.resName = residue
          if name:
            coordination_atom.name = name
          print("-"*80)
          print(metal.Display())
          print(coordination_atom.Display())
          rc = run(metal, coordination_atom)
          print(rc)
          if rc!=results[i]:
            print(i, rc, results[i])
            assert 0
          i+=1
  print("OK")

if __name__=="__main__":
  if len(sys.argv)>1:
    print(get_default_bondlength(*tuple(sys.argv[1:])))
  else:
    for atoms in [("At", "At"),
                  ("At", "H"),
        ]:
      print(atoms, get_default_bondlength(*atoms))
